/*
 * (c) 2003-2023 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.test.config.registry;

import static org.mule.test.allure.AllureConstants.RegistryFeature.REGISTRY;
import static org.mule.test.allure.AllureConstants.RegistryFeature.ObjectRegistrationStory.OBJECT_REGISTRATION;

import static java.util.Arrays.asList;
import static java.util.Collections.emptySet;
import static java.util.stream.Collectors.toSet;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.collection.IsEmptyCollection.empty;

import org.mule.functional.junit4.MuleArtifactFunctionalTestCase;
import org.mule.runtime.api.artifact.Registry;
import org.mule.runtime.api.cluster.ClusterService;
import org.mule.runtime.api.component.ConfigurationProperties;
import org.mule.runtime.api.component.location.ConfigurationComponentLocator;
import org.mule.runtime.api.config.FeatureFlaggingService;
import org.mule.runtime.api.connectivity.ConnectivityTestingService;
import org.mule.runtime.api.deployment.management.ComponentInitialStateManager;
import org.mule.runtime.api.el.ExpressionLanguage;
import org.mule.runtime.api.exception.ErrorTypeRepository;
import org.mule.runtime.api.lifecycle.InitialisationException;
import org.mule.runtime.api.lock.LockFactory;
import org.mule.runtime.api.memory.management.MemoryManagementService;
import org.mule.runtime.api.metadata.MetadataService;
import org.mule.runtime.api.notification.NotificationDispatcher;
import org.mule.runtime.api.notification.NotificationListenerRegistry;
import org.mule.runtime.api.profiling.ProfilingService;
import org.mule.runtime.api.scheduler.SchedulerConfig;
import org.mule.runtime.api.scheduler.SchedulerPoolsConfigFactory;
import org.mule.runtime.api.serialization.ObjectSerializer;
import org.mule.runtime.api.service.Service;
import org.mule.runtime.api.store.ObjectStore;
import org.mule.runtime.api.store.ObjectStoreManager;
import org.mule.runtime.api.time.TimeSupplier;
import org.mule.runtime.api.transformation.TransformationService;
import org.mule.runtime.api.tx.DataSourceDecorator;
import org.mule.runtime.api.util.ResourceLocator;
import org.mule.runtime.api.value.ValueProviderService;
import org.mule.runtime.config.api.LazyComponentInitializer;
import org.mule.runtime.core.api.MuleContext;
import org.mule.runtime.core.api.config.ConfigurationException;
import org.mule.runtime.core.api.config.MuleConfiguration;
import org.mule.runtime.core.api.connector.ConnectionManager;
import org.mule.runtime.core.api.context.notification.FlowTraceManager;
import org.mule.runtime.core.api.context.notification.ServerNotificationHandler;
import org.mule.runtime.core.api.data.sample.SampleDataService;
import org.mule.runtime.core.api.event.EventContextService;
import org.mule.runtime.core.api.execution.ExceptionContextProvider;
import org.mule.runtime.core.api.extension.ExtensionManager;
import org.mule.runtime.core.api.lifecycle.LifecycleManager;
import org.mule.runtime.core.api.management.stats.AllStatistics;
import org.mule.runtime.core.api.management.stats.ProcessingTimeWatcher;
import org.mule.runtime.core.api.policy.OperationPolicyParametersTransformer;
import org.mule.runtime.core.api.policy.PolicyProvider;
import org.mule.runtime.core.api.policy.SourcePolicyParametersTransformer;
import org.mule.runtime.core.api.security.SecurityManager;
import org.mule.runtime.core.api.streaming.StreamingManager;
import org.mule.runtime.core.api.transformer.DataTypeConversionResolver;
import org.mule.runtime.core.api.transformer.Transformer;
import org.mule.runtime.core.api.util.StreamCloserService;
import org.mule.runtime.core.api.util.queue.QueueManager;
import org.mule.runtime.core.privileged.el.GlobalBindingContextProvider;
import org.mule.runtime.core.privileged.exception.ErrorTypeLocator;
import org.mule.runtime.core.privileged.registry.ObjectProcessor;
import org.mule.runtime.core.privileged.transformer.TransformersRegistry;
import org.mule.runtime.extension.api.client.ExtensionsClient;
import org.mule.runtime.feature.api.management.FeatureFlaggingManagementService;
import org.mule.runtime.metadata.api.cache.MetadataCacheIdGeneratorFactory;
import org.mule.runtime.module.artifact.api.classloader.ClassLoaderRepository;
import org.mule.runtime.policy.api.OperationPolicyPointcutParametersFactory;
import org.mule.runtime.policy.api.SourcePolicyPointcutParametersFactory;
import org.mule.sdk.compatibility.api.utils.ForwardCompatibilityHelper;
import org.mule.test.runner.RunnerDelegateTo;

import com.mulesoft.mule.runtime.bti.api.jms.ConnectionFactoryDecorator;
import com.mulesoft.mule.runtime.module.batch.api.BatchManager;

import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Stream;

import javax.inject.Inject;

import org.junit.Test;
import org.junit.runners.Parameterized;
import org.junit.runners.Parameterized.Parameter;
import org.junit.runners.Parameterized.Parameters;

import io.qameta.allure.Feature;
import io.qameta.allure.Story;

/**
 * Tests to ensure that the process to add/remove things from the registry is properly followed.
 */
@Feature(REGISTRY)
@Story(OBJECT_REGISTRATION)
@RunnerDelegateTo(Parameterized.class)
public class RegistryContentsTestCase extends MuleArtifactFunctionalTestCase {

  @Parameters(name = "lazyInit:{0}; addToolingObjectsToRegistry: {1}")
  public static List<Boolean[]> lazyInit() {
    return asList(new Boolean[] {false, false},
                  new Boolean[] {false, true},
                  new Boolean[] {true, false},
                  new Boolean[] {true, true});
  }

  private static final Set<Class> API_REGISTRY_ENTRY_TYPES = Stream
      .of(Service.class,
          Registry.class,
          ObjectSerializer.class,
          SchedulerConfig.class,
          SchedulerPoolsConfigFactory.class,
          ExpressionLanguage.class,
          ConfigurationComponentLocator.class,
          ObjectStore.class,
          ObjectStoreManager.class,
          org.mule.sdk.api.store.ObjectStoreManager.class,
          ConfigurationProperties.class,
          DataSourceDecorator.class,
          FeatureFlaggingService.class,
          NotificationDispatcher.class,
          ExtensionsClient.class,
          LockFactory.class,
          ErrorTypeRepository.class,
          TransformationService.class,
          MemoryManagementService.class,
          ComponentInitialStateManager.class,
          FeatureFlaggingManagementService.class,
          TimeSupplier.class,
          NotificationListenerRegistry.class,
          ProfilingService.class,
          ClusterService.class,
          SourcePolicyPointcutParametersFactory.class,
          OperationPolicyPointcutParametersFactory.class,
          ForwardCompatibilityHelper.class)
      .collect(toSet());

  private static final Set<Class> API_TOOLING_REGISTRY_ENTRY_TYPES = Stream
      .of(ValueProviderService.class,
          MetadataService.class,
          ConnectivityTestingService.class)
      .collect(toSet());

  private static final Set<Class> CORE_API_REGISTRY_ENTRY_TYPES = Stream
      .of(EventContextService.class,
          SecurityManager.class,
          ServerNotificationHandler.class,
          ClassLoaderRepository.class,
          ConnectionManager.class,
          Transformer.class,
          LifecycleManager.class,
          LazyComponentInitializer.class,
          QueueManager.class,
          ExceptionContextProvider.class,
          BatchManager.class,
          ExtensionManager.class,
          AllStatistics.class,
          FlowTraceManager.class,
          StreamingManager.class,
          StreamCloserService.class,
          PolicyProvider.class,
          OperationPolicyParametersTransformer.class,
          SourcePolicyParametersTransformer.class,
          ResourceLocator.class,
          ProcessingTimeWatcher.class,
          ConnectionFactoryDecorator.class,
          DataTypeConversionResolver.class,
          MuleContext.class,
          MuleConfiguration.class)
      .collect(toSet());

  private static final Set<Class> CORE_TOOLING_API_REGISTRY_ENTRY_TYPES = Stream
      .of(SampleDataService.class,
          MetadataCacheIdGeneratorFactory.class)
      .collect(toSet());

  private static final Set<Class> PRIVILEGED_REGISTRY_ENTRY_TYPES = Stream
      .of(GlobalBindingContextProvider.class,
          ErrorTypeLocator.class,
          TransformersRegistry.class,
          ObjectProcessor.class)
      .collect(toSet());

  private static final Set<String> INTERNAL_REGISTRY_ENTRY_TYPE_NAMES = Stream
      .of("org.mule.runtime.config.internal.bean.ServerNotificationManagerConfigurator",
          "org.mule.runtime.core.internal.connection.ConnectivityTesterFactory",
          "org.mule.runtime.core.internal.el.ExpressionLanguageAdaptor",
          "org.mule.runtime.core.internal.el.mvel.ExpressionLanguageExtension",
          "org.mule.runtime.core.internal.execution.MessageProcessingManager",
          "org.mule.runtime.core.internal.interception.InterceptorManager",
          "org.mule.runtime.core.internal.lock.LockProvider",
          "org.mule.runtime.core.internal.policy.PolicyManager",
          "org.mule.runtime.core.internal.registry.TransformerResolver",
          "org.mule.runtime.core.internal.streaming.StreamingGhostBuster",
          "org.mule.runtime.core.internal.transaction.TransactionFactoryLocator",
          "org.mule.runtime.core.internal.type.catalog.DefaultArtifactTypeLoader",
          "org.mule.runtime.module.extension.internal.runtime.connectivity.ExtensionConnectionSupplier",
          "org.mule.runtime.module.extension.internal.runtime.connectivity.oauth.OAuthHandler",
          "org.mule.runtime.module.extension.internal.runtime.execution.executor.MethodExecutorGenerator",
          "org.mule.runtime.module.extension.internal.util.ReflectionCache",
          "com.mulesoft.mule.runtime.module.batch.engine.BatchEngine",
          "com.mulesoft.mule.runtime.module.batch.engine.BatchJobInstanceStore",
          "com.mulesoft.mule.runtime.module.batch.engine.queue.BatchQueueManager",
          "com.mulesoft.mule.runtime.module.batch.internal.engine.BatchLockFactory",
          "com.mulesoft.mule.runtime.module.batch.internal.engine.threading.BatchWorkManager",
          "com.mulesoft.mule.runtime.tracking.internal.config.Initialisation",
          "org.springframework.beans.factory.config.BeanPostProcessor",
          "org.springframework.context.LifecycleProcessor",
          "org.springframework.context.MessageSource",
          "org.springframework.context.event.ApplicationEventMulticaster",
          "org.springframework.core.env.Environment",
          "org.springframework.core.metrics.ApplicationStartup",
          "org.mule.runtime.tracer.impl.CoreEventTracer",
          "org.mule.runtime.tracer.impl.span.factory.ExecutionSpanFactory",
          "org.mule.runtime.tracer.impl.exporter.OpenTelemetrySpanExporterFactory",
          "org.mule.runtime.tracer.impl.exporter.optel.config.OpenTelemetryAutoConfigurableSpanExporterConfiguration")
      .collect(toSet());

  private static final Set<String> INTERNAL_TOOLING_REGISTRY_ENTRY_TYPE_NAMES = Stream
      .of("org.mule.runtime.core.internal.metadata.cache.MetadataCacheManager")
      .collect(toSet());

  @Parameter(0)
  public boolean lazyInit;

  @Parameter(1)
  public boolean addToolingObjectsToRegistry;

  @Override
  protected String[] getConfigFiles() {
    return new String[0];
  }

  @Override
  public boolean enableLazyInit() {
    return lazyInit;
  }

  @Override
  public boolean addToolingObjectsToRegistry() {
    return addToolingObjectsToRegistry;
  }

  @Inject
  private Registry registry;

  @Test
  public void validateRuntimeRegistryNoAdditionalEntries() throws InitialisationException, ConfigurationException {
    Collection<Object> allRuntimeRegistryEntries = registry.lookupAllByType(Object.class);

    Set<ClassLoader> nonContainerClassLoaders = allRuntimeRegistryEntries.stream()
        .filter(t -> t.getClass().getClassLoader() != null)
        .filter(t -> {
          String classLoaderId = t.getClass().getClassLoader().toString();
          return classLoaderId
              .startsWith("org.mule.runtime.module.artifact.internal.classloader.MulePluginClassLoader[Region/plugin/")
              || classLoaderId
                  .startsWith("org.mule.runtime.module.artifact.activation.internal.classloader.MuleApplicationClassLoader[app]");
        })
        .map(t -> t.getClass().getClassLoader())
        .collect(toSet());

    Set<Object> unknownRegistryEntries = allRuntimeRegistryEntries.stream()
        .filter(o -> o.getClass().getClassLoader() != null
            && !nonContainerClassLoaders.contains(o.getClass().getClassLoader()))
        .filter(o -> API_REGISTRY_ENTRY_TYPES.stream()
            .noneMatch(t -> t.isAssignableFrom(o.getClass())))
        .filter(o -> CORE_API_REGISTRY_ENTRY_TYPES.stream()
            .noneMatch(t -> t.isAssignableFrom(o.getClass())))
        .filter(o -> !addToolingObjectsToRegistry
            && CORE_TOOLING_API_REGISTRY_ENTRY_TYPES.stream()
                .noneMatch(t -> t.isAssignableFrom(o.getClass())))
        .filter(o -> PRIVILEGED_REGISTRY_ENTRY_TYPES.stream()
            .noneMatch(t -> t.isAssignableFrom(o.getClass())))
        .filter(o -> INTERNAL_REGISTRY_ENTRY_TYPE_NAMES.stream()
            .noneMatch(t -> getAllClassesAndInterfacesInHierarchyNames(o.getClass()).contains(t)))
        .filter(o -> !addToolingObjectsToRegistry
            && INTERNAL_TOOLING_REGISTRY_ENTRY_TYPE_NAMES.stream()
                .noneMatch(t -> getAllClassesAndInterfacesInHierarchyNames(o.getClass()).contains(t)))
        .collect(toSet());

    assertThat("Please validate the objects that were added to the registry."
        + " (ref: https://docs.google.com/document/d/1XtwQCGoTRjddtNvw1jJzzmDsQUJ7YHM8O4dv7JjKHkc)",
               unknownRegistryEntries, empty());
  }

  @Test
  public void validateRuntimeRegistryAllRequiredEntries() throws InitialisationException, ConfigurationException {
    Collection<Object> allRuntimeRegistryEntries = registry.lookupAllByType(Object.class);

    assertThat("Please validate the objects that were removed from the registry."
        + " (ref: https://docs.google.com/document/d/1XtwQCGoTRjddtNvw1jJzzmDsQUJ7YHM8O4dv7JjKHkc)",
               API_REGISTRY_ENTRY_TYPES.stream()
                   .filter(t -> allRuntimeRegistryEntries.stream()
                       .noneMatch(o -> t.isAssignableFrom(o.getClass())))
                   .collect(toSet()),
               empty());
    if (addToolingObjectsToRegistry) {
      assertThat("Please validate the objects that were removed from the registry."
          + " (ref: https://docs.google.com/document/d/1XtwQCGoTRjddtNvw1jJzzmDsQUJ7YHM8O4dv7JjKHkc)",
                 API_TOOLING_REGISTRY_ENTRY_TYPES.stream()
                     .filter(t -> allRuntimeRegistryEntries.stream()
                         .noneMatch(o -> t.isAssignableFrom(o.getClass())))
                     .collect(toSet()),
                 empty());
    }
    assertThat("Please validate the objects that were removed from the registry."
        + " (ref: https://docs.google.com/document/d/1XtwQCGoTRjddtNvw1jJzzmDsQUJ7YHM8O4dv7JjKHkc)",
               CORE_API_REGISTRY_ENTRY_TYPES.stream()
                   .filter(t -> allRuntimeRegistryEntries.stream()
                       .noneMatch(o -> t.isAssignableFrom(o.getClass())))
                   .collect(toSet()),
               empty());
    if (addToolingObjectsToRegistry) {
      assertThat("Please validate the objects that were removed from the registry."
          + " (ref: https://docs.google.com/document/d/1XtwQCGoTRjddtNvw1jJzzmDsQUJ7YHM8O4dv7JjKHkc)",
                 CORE_TOOLING_API_REGISTRY_ENTRY_TYPES.stream()
                     .filter(t -> allRuntimeRegistryEntries.stream()
                         .noneMatch(o -> t.isAssignableFrom(o.getClass())))
                     .collect(toSet()),
                 empty());
    }
    assertThat("Please validate the objects that were removed from the registry."
        + " (ref: https://docs.google.com/document/d/1XtwQCGoTRjddtNvw1jJzzmDsQUJ7YHM8O4dv7JjKHkc)",
               PRIVILEGED_REGISTRY_ENTRY_TYPES.stream()
                   .filter(t -> allRuntimeRegistryEntries.stream()
                       .noneMatch(o -> t.isAssignableFrom(o.getClass())))
                   .collect(toSet()),
               empty());
  }

  private Collection<String> getAllClassesAndInterfacesInHierarchyNames(Class cls) {
    return getAllClassesAndInterfacesInHierarchy(cls).stream()
        .map(c -> c.getName())
        .collect(toSet());
  }

  private Collection<Class> getAllClassesAndInterfacesInHierarchy(Class cls) {
    if (cls == null) {
      return emptySet();
    }

    Set<Class> hierarchyItems = new HashSet<>();
    hierarchyItems.add(cls);

    Class[] interfaces = cls.getInterfaces();
    for (Class iface : interfaces) {
      hierarchyItems.addAll(getAllClassesAndInterfacesInHierarchy(iface));
    }
    hierarchyItems.addAll(getAllClassesAndInterfacesInHierarchy(cls.getSuperclass()));

    return hierarchyItems;
  }
}
