/*
 * (c) 2003-2023 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.test.module.cache;

import static java.util.Collections.singletonList;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.core.Is.is;
import static org.mule.runtime.api.component.ComponentIdentifier.buildFromStringRepresentation;
import static org.mule.runtime.api.component.TypedComponentIdentifier.builder;
import static org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType.CHAIN;
import static org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType.FLOW;
import static org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType.OPERATION;
import static org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType.SCOPE;
import static org.mule.runtime.api.notification.MessageProcessorNotification.MESSAGE_PROCESSOR_PRE_INVOKE;
import static org.mule.runtime.config.api.dsl.CoreDslConstants.FLOW_IDENTIFIER;
import static org.mule.test.allure.AllureConstants.ConfigurationComponentLocatorFeature.CONFIGURATION_COMPONENT_LOCATOR;
import static org.mule.test.allure.AllureConstants.ConfigurationComponentLocatorFeature.ConfigurationComponentLocationStory.COMPONENT_LOCATION;

import org.mule.functional.junit4.MuleArtifactFunctionalTestCase;
import org.mule.runtime.api.component.TypedComponentIdentifier;
import org.mule.runtime.api.component.location.ComponentLocation;
import org.mule.runtime.api.component.location.Location;
import org.mule.runtime.api.notification.IntegerAction;
import org.mule.runtime.api.notification.MessageProcessorNotification;
import org.mule.runtime.api.notification.MessageProcessorNotificationListener;
import org.mule.runtime.ast.api.ArtifactAst;
import org.mule.runtime.ast.api.xml.AstXmlParser;
import org.mule.runtime.dsl.api.component.config.DefaultComponentLocation;
import org.mule.tck.core.context.notification.AbstractNotificationLogger;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Optional;
import java.util.OptionalInt;

import org.junit.After;
import org.junit.Assert;
import org.junit.Test;

import com.google.common.collect.ImmutableList;

import io.qameta.allure.Description;
import io.qameta.allure.Feature;
import io.qameta.allure.Issue;
import io.qameta.allure.Story;

@Feature(CONFIGURATION_COMPONENT_LOCATOR)
@Story(COMPONENT_LOCATION)
public class CacheComponentPathTestCase extends MuleArtifactFunctionalTestCase {

  private static final Optional<String> CONFIG_FILE_NAME = of("module/cache/flows-using-smart-connector.xml");

  private static final String COLON_SEPARATOR = ":";
  private static final String MODULE_SIMPLE_XML = "module-using-core.xml";
  private static final String MODULE_SIMPLE_NAMESPACE_IN_APP = "module-using-core";

  private static final Optional<String> MODULE_SIMPLE_FILE_NAME = of(MODULE_SIMPLE_XML);

  private static final String FLOW_WITH_CACHE_NAME = "flowWithCacheScope";
  private static final String SET_PAYLOAD_VALUE_NAME = "echo-set-payload";

  private static final String MODULE_BODY_NAMESPACE_PREFIX = "module";
  private static final Optional<TypedComponentIdentifier> MODULE_BODY =
      getModuleBodyIdentifier();

  private static final Optional<TypedComponentIdentifier> MODULE_SET_PAYLOAD_VALUE =
      getModuleOperationIdentifier(MODULE_SIMPLE_NAMESPACE_IN_APP, SET_PAYLOAD_VALUE_NAME);

  private static final Optional<TypedComponentIdentifier> FLOW_TYPED_COMPONENT_IDENTIFIER =
      of(TypedComponentIdentifier.builder().identifier(FLOW_IDENTIFIER).type(FLOW).build());

  private static final DefaultComponentLocation OPERATION_SET_PAYLOAD =
      getModuleOperationLocation(SET_PAYLOAD_VALUE_NAME, MODULE_BODY, MODULE_SIMPLE_FILE_NAME,
                                 26);
  private static final DefaultComponentLocation FLOW_WITH_CACHE = getFlowLocation(FLOW_WITH_CACHE_NAME, 17);

  private static final Optional<TypedComponentIdentifier> SET_PAYLOAD =
      of(TypedComponentIdentifier.builder().identifier(buildFromStringRepresentation("mule:set-payload"))
          .type(OPERATION).build());

  private static final Optional<TypedComponentIdentifier> CACHE =
      of(TypedComponentIdentifier.builder().identifier(buildFromStringRepresentation("ee:cache"))
          .type(SCOPE).build());

  protected static final ProcessorNotificationStore listener = new ProcessorNotificationStore();

  @Override
  protected String getConfigFile() {
    return CONFIG_FILE_NAME.get();
  }

  private static Optional<TypedComponentIdentifier> getModuleBodyIdentifier() {
    return Optional.of(builder()
        .identifier(buildFromStringRepresentation(MODULE_BODY_NAMESPACE_PREFIX + COLON_SEPARATOR + "body"))
        .type(CHAIN).build());
  }

  private static DefaultComponentLocation getFlowLocation(final String flowName, final int flowLineNumber) {
    return new DefaultComponentLocation(of(flowName), singletonList(
                                                                    new DefaultComponentLocation.DefaultLocationPart(flowName,
                                                                                                                     FLOW_TYPED_COMPONENT_IDENTIFIER,
                                                                                                                     CONFIG_FILE_NAME,
                                                                                                                     OptionalInt
                                                                                                                         .of(flowLineNumber),
                                                                                                                     OptionalInt
                                                                                                                         .of(5))));
  }

  private static DefaultComponentLocation getModuleOperationLocation(final String operationName,
                                                                     final Optional<TypedComponentIdentifier> operationIdentifier,
                                                                     final Optional<String> moduleFilename,
                                                                     final int operationLineNumber) {
    return new DefaultComponentLocation(of(operationName),
                                        singletonList(new DefaultComponentLocation.DefaultLocationPart(operationName,
                                                                                                       operationIdentifier,
                                                                                                       moduleFilename,
                                                                                                       OptionalInt
                                                                                                           .of(operationLineNumber),
                                                                                                       OptionalInt.of(9))));
  }

  private static Optional<TypedComponentIdentifier> getModuleOperationIdentifier(final String namespace,
                                                                                 final String identifier) {
    return of(TypedComponentIdentifier.builder()
        .identifier(buildFromStringRepresentation(namespace + COLON_SEPARATOR + identifier))
        .type(OPERATION).build());
  }

  @Override
  protected void doSetUp() throws Exception {
    super.doSetUp();
    listener.setLogSingleNotification(true);
    muleContext.getNotificationManager().addListener(listener);
  }

  @After
  public void clearNotifications() {
    if (listener != null) {
      listener.getNotifications().clear();
    }
  }

  @Description("Smart Connector inside a cache scope")
  @Issue("MULE-16285")
  @Test
  public void flowWithCacheScope() throws Exception {
    flowRunner(FLOW_WITH_CACHE_NAME).run();

    DefaultComponentLocation firstComponentLocation = FLOW_WITH_CACHE
        .appendLocationPart("processors", empty(), empty(), OptionalInt.empty(), OptionalInt.empty())
        .appendLocationPart("0", CACHE, CONFIG_FILE_NAME, OptionalInt.of(18), OptionalInt.of(9));

    assertNextProcessorLocationIs(firstComponentLocation);

    assertNextProcessorLocationIs(firstComponentLocation
        .appendLocationPart("processors", empty(), empty(), OptionalInt.empty(), OptionalInt.empty())
        .appendLocationPart("0", MODULE_SET_PAYLOAD_VALUE, CONFIG_FILE_NAME, OptionalInt.of(19),
                            OptionalInt.of(13)));

    assertNextProcessorLocationIs(OPERATION_SET_PAYLOAD
        .appendLocationPart("processors", empty(), empty(), OptionalInt.empty(), OptionalInt.empty())
        .appendLocationPart("0", SET_PAYLOAD, MODULE_SIMPLE_FILE_NAME, OptionalInt.of(27), OptionalInt.of(13)));
    assertNoNextProcessorNotification();
  }

  @Description("Validate component location created from extension models")
  @Test
  public void validateComponentLocationCreatedFromExtensionModelsWithoutUsingParsers() throws Exception {
    AstXmlParser xmlToAstParser = AstXmlParser.builder()
        .withExtensionModels(muleContext.getExtensionManager().getExtensions())
        .withSchemaValidationsDisabled()
        .build();

    ArtifactAst toolingApplicationModel =
        xmlToAstParser.parse(getClass().getClassLoader().getResource(CONFIG_FILE_NAME.get()).toURI());

    List<String> componentLocations = new ArrayList<>();
    toolingApplicationModel.recursiveStream().forEach(componentModel -> {
      final String componentIdentifierName = componentModel.getIdentifier().getName();
      if (componentIdentifierName.equals("notification") || componentIdentifierName.equals("notifications")) {
        return;
      }
      final ComponentLocation componentLocation = componentModel.getLocation();
      if (componentLocation != null) {
        componentLocations.add(componentLocation.getLocation());
      }
    });

    assertThat(ImmutableList.builder()
        .add(Location.builder().globalName(FLOW_WITH_CACHE_NAME).build().toString())
        .add(Location.builder().globalName(FLOW_WITH_CACHE_NAME).addProcessorsPart().addIndexPart(0)
            .build().toString())
        .add(Location.builder().globalName(FLOW_WITH_CACHE_NAME).addProcessorsPart().addIndexPart(0)
            .addProcessorsPart().addIndexPart(0).build().toString())
        .build(), is(componentLocations));
  }


  // TODO: MULE-17049
  private void assertNextProcessorLocationIs(DefaultComponentLocation componentLocation) {
    assertThat(listener.getNotifications().isEmpty(), is(false));
    MessageProcessorNotification processorNotification =
        listener.getNotifications().get(0);
    listener.getNotifications().remove(0);
    assertThat(processorNotification.getComponent().getLocation().getLocation(), is(componentLocation.getLocation()));
    assertThat(processorNotification.getComponent().getLocation(), is(componentLocation));
  }

  private void assertNoNextProcessorNotification() {
    Iterator<MessageProcessorNotification> iterator = listener.getNotifications().iterator();
    Assert.assertThat(iterator.hasNext(), is(false));
  }

  public static class ProcessorNotificationStore extends AbstractNotificationLogger<MessageProcessorNotification>
      implements MessageProcessorNotificationListener<MessageProcessorNotification> {

    boolean logSingleNotification = false;

    @Override
    public boolean isBlocking() {
      return false;
    }

    @Override
    public synchronized void onNotification(MessageProcessorNotification notification) {
      if (!logSingleNotification || new IntegerAction(MESSAGE_PROCESSOR_PRE_INVOKE).equals(notification.getAction())) {
        super.onNotification(notification);
      }
    }

    public void setLogSingleNotification(boolean logSingleNotification) {
      this.logSingleNotification = logSingleNotification;
    }
  }
}
