/*
 * (c) 2003-2023 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.test.module.http;

import static org.apache.commons.io.FileUtils.write;
import static org.hamcrest.CoreMatchers.equalTo;
import static org.hamcrest.CoreMatchers.instanceOf;
import static org.hamcrest.CoreMatchers.is;
import static org.hamcrest.CoreMatchers.notNullValue;
import static org.hamcrest.text.IsEqualIgnoringWhiteSpace.equalToIgnoringWhiteSpace;
import static org.junit.Assert.assertThat;
import static org.mule.runtime.http.api.HttpConstants.Method.GET;

import org.mule.extension.ftp.DefaultFtpTestHarness;
import org.mule.functional.junit4.MuleArtifactFunctionalTestCase;
import org.mule.runtime.core.api.util.IOUtils;
import org.mule.runtime.http.api.HttpService;
import org.mule.runtime.http.api.domain.message.request.HttpRequest;
import org.mule.runtime.http.api.domain.message.response.HttpResponse;
import org.mule.service.http.TestHttpClient;
import org.mule.tck.junit4.rule.DynamicPort;
import org.mule.tck.junit4.rule.SystemProperty;

import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;

import java.io.File;
import java.io.InputStream;
import java.net.URL;
import java.util.List;
import java.util.Map;


public class HttpRequestAndTransformWhileStreamingTestCase extends MuleArtifactFunctionalTestCase {

  private static final String TEST_JSON_FILE = "streaming/patients.json";
  private static final String TEST_XML_FILE = "streaming/accounts.xml";

  @ClassRule
  public static TemporaryFolder temporaryFolder = new TemporaryFolder();

  @Rule
  public SystemProperty workingDir = new SystemProperty("workingDir", getWorkingDirPath());

  @Rule
  public DynamicPort httpPort = new DynamicPort("httpPort");

  @Rule
  public final DefaultFtpTestHarness testHarness = new DefaultFtpTestHarness();

  @Rule
  public TestHttpClient httpClient = new TestHttpClient.Builder(getService(HttpService.class)).build();

  @Rule
  public SystemProperty fileWorkingDir = new SystemProperty("fileWorkingDir", temporaryFolder.getRoot().getAbsolutePath());

  private String getWorkingDirPath() {
    URL url = HttpRequestAndTransformWhileStreamingTestCase.class.getClassLoader().getResource(TEST_JSON_FILE);
    assertThat("patients file not found", url, is(notNullValue()));
    try {
      return new File(url.toURI()).getParentFile().getAbsolutePath();
    } catch (Exception e) {
      throw new RuntimeException(e);
    }
  }

  @Override
  protected String getConfigFile() {
    return "streaming/http-request-transform.xml";
  }

  @Override
  protected void doSetUp() throws Exception {
    super.doSetUp();
    putPatientsFileInFtp();
  }

  private void putPatientsFileInFtp() throws Exception {
    testHarness.makeDir("streaming");
    try (InputStream in =
        HttpRequestAndTransformWhileStreamingTestCase.class.getClassLoader().getResourceAsStream(TEST_JSON_FILE)) {
      assertThat("test file not found", in, is(notNullValue()));
      testHarness.write(TEST_JSON_FILE, IOUtils.toString(in));
    }
  }

  @Test
  public void inMemoryStreaming() throws Exception {
    transformAndAssert("inMemoryStreaming");
  }

  @Test
  public void fileStreaming() throws Exception {
    transformAndAssert("fileStoreStreaming");
  }

  @Test
  public void noStreaming() throws Exception {
    transformAndAssert("noStreaming");
  }

  @Test
  public void proxyWithInMemoryStreaming() throws Exception {
    doProxy("proxyInMemory");
  }

  @Test
  public void proxyWithFileStoreStreaming() throws Exception {
    doProxy("proxyFileStore");
  }

  @Test
  public void proxyWithoutStreaming() throws Exception {
    doProxy("proxyNoStreaming");
  }

  @Test
  public void transformXmlInMemory() throws Exception {
    doTransformXML("xmlTransformInMemory");
  }

  @Test
  public void transformXmlFileStore() throws Exception {
    doTransformXML("xmlTransformFileStore");
  }

  @Test
  public void transformXmlNoStream() throws Exception {
    doTransformXML("xmlTransformNoStreaming");
  }

  private void doTransformXML(String flowName) throws Exception {
    try (InputStream in =
        HttpRequestAndTransformWhileStreamingTestCase.class.getClassLoader().getResourceAsStream(TEST_XML_FILE)) {
      assertThat("test file not found", in, is(notNullValue()));
      write(new File(temporaryFolder.getRoot(), "accounts.xml"), IOUtils.toString(in));
    }

    String actual = (String) flowRunner(flowName).run().getMessage().getPayload().getValue();
    assertThat(actual,
               equalToIgnoringWhiteSpace(IOUtils.toString(HttpRequestAndTransformWhileStreamingTestCase.class.getClassLoader()
                   .getResourceAsStream("streaming/account-ids.list"))));
  }

  private void doProxy(String path) throws Exception {
    HttpRequest request = HttpRequest.builder()
        .uri(String.format("http://localhost:%d/%s", httpPort.getNumber(), path))
        .method(GET)
        .build();
    HttpResponse response = httpClient.send(request, 999999999, false, null);
    assertThat(IOUtils.toString(response.getEntity().getContent()), equalTo("Proxy baby"));
  }

  private void transformAndAssert(String flowName) throws Exception {
    Object value = flowRunner(flowName).run().getMessage().getPayload().getValue();

    assertThat(value, is(instanceOf(List.class)));
    Object firstItem = ((List) value).get(0);
    assertThat(firstItem, is(instanceOf(Map.class)));
    Map<String, Object> map = (Map<String, Object>) firstItem;
    assertThat(map.get("LAST_NAME"), equalTo("Baddeley"));
    assertThat(map.get("FIRST_NAME"), equalTo("Chris"));
  }

}
