/*
 * (c) 2003-2023 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.test.module.streaming;

import static org.apache.commons.io.FileUtils.readFileToString;
import static org.hamcrest.CoreMatchers.notNullValue;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.containsString;
import static org.hamcrest.core.Is.is;
import static org.hamcrest.text.IsEqualIgnoringWhiteSpace.equalToIgnoringWhiteSpace;
import static org.mule.runtime.core.api.util.IOUtils.closeQuietly;
import static org.mule.runtime.core.api.util.IOUtils.getResourceAsUrl;
import static org.mule.runtime.http.api.HttpConstants.HttpStatus.INTERNAL_SERVER_ERROR;
import static org.mule.runtime.http.api.HttpConstants.HttpStatus.OK;
import static org.mule.test.allure.AllureConstants.HttpFeature.HTTP_EXTENSION;
import static org.mule.test.allure.AllureConstants.IntegrationTestsFeature.INTEGRATIONS_TESTS;
import static org.mule.test.allure.AllureConstants.StreamingFeature.STREAMING;
import static org.mule.test.allure.AllureConstants.TransformMessageFeature.TRANSFORM_MESSAGE;

import io.qameta.allure.Description;

import io.qameta.allure.Feature;
import io.qameta.allure.Stories;
import io.qameta.allure.Story;
import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.apache.http.entity.mime.MultipartEntityBuilder;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.junit.ClassRule;
import org.junit.Ignore;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;
import org.mule.functional.junit4.MuleArtifactFunctionalTestCase;
import org.mule.runtime.core.api.util.IOUtils;
import org.mule.tck.junit4.rule.DynamicPort;
import org.mule.tck.junit4.rule.SystemProperty;
import org.mule.tck.probe.JUnitLambdaProbe;
import org.mule.tck.probe.PollingProber;

import java.io.File;
import java.io.InputStream;
import java.nio.file.Paths;

@Feature(INTEGRATIONS_TESTS)
@Stories({@Story(HTTP_EXTENSION), @Story(STREAMING), @Story(TRANSFORM_MESSAGE)})
// @RunnerDelegateTo(FlakinessDetectorTestRunner.class)
public class StreamingTransformationsTestCase extends MuleArtifactFunctionalTestCase {

  private static final ContentType JSON_CONTENT = ContentType.APPLICATION_JSON;
  private static final ContentType CSV_CONTENT = ContentType.create("APPLICATION/csv");

  private static final String CSV_INPUT_FILE = "streaming/MOCK_DATA_small.csv";
  private static final String JSON_INPUT_FILE = "streaming/MOCK_DATA_small.json";
  private static final String JSON_RESULTS_FILE = "streaming/results.json";

  private static final String BUFFER_EXCEEDED_MESSAGE = "Buffer has exceeded its maximum size of 20480";

  private static final int TIMEOUT_MILLIS = 30000;
  private static final int POLL_DELAY_MILLIS = 1000;

  @ClassRule
  public static TemporaryFolder temporaryFolder = new TemporaryFolder();

  @Rule
  public DynamicPort httpPort = new DynamicPort("http.port");

  @Rule
  public SystemProperty workingDir = new SystemProperty("working.dir", temporaryFolder.getRoot().getAbsolutePath());

  private CloseableHttpClient client;

  protected void doSetUp() throws Exception {
    super.doSetUp();
    client = HttpClients.createDefault();
  }

  @Override
  protected void doTearDown() throws Exception {
    super.doTearDown();
    if (client != null) {
      closeQuietly(client);
    }
  }

  @Override
  protected String getConfigFile() {
    return "streaming/streaming-transformations.xml";
  }

  @Test
  @Description("Receive a CSV file via a Multipart POST using a repeatable in memory stream and echo it.")
  public void testRepeatableMemoryStreamSmallFile() throws Exception {
    File csvInput = new File(getResourceAsUrl(CSV_INPUT_FILE, StreamingTransformationsTestCase.class).toURI());
    HttpPost request = buildMultipartRequest("testRepeatableMemoryStreamSmallFile", CSV_CONTENT, csvInput);
    HttpResponse response = client.execute(request);

    assertThat(response.getStatusLine().getStatusCode(), is(OK.getStatusCode()));

    String content = IOUtils.toString(response.getEntity().getContent());
    String fileContent = getFileContent(CSV_INPUT_FILE);
    assertThat(content, containsString(fileContent));
  }

  @Test
  @Description("Receive a CSV file via a Multipart POST using a repeatable in memory stream with an insufficient buffer.")
  public void testRepeatableMemoryStreamSmallFileBufferOverflow() throws Exception {
    File csvInput = new File(getResourceAsUrl(CSV_INPUT_FILE, StreamingTransformationsTestCase.class).toURI());
    HttpPost request = buildMultipartRequest("testRepeatableMemoryStreamSmallFileBufferOverflow", CSV_CONTENT, csvInput);
    HttpResponse response = client.execute(request);

    assertThat(response.getStatusLine().getStatusCode(), is(INTERNAL_SERVER_ERROR.getStatusCode()));

    String content = IOUtils.toString(response.getEntity().getContent());
    assertThat(content, equalToIgnoringWhiteSpace(BUFFER_EXCEEDED_MESSAGE));
  }

  @Test
  @Description("Receive a CSV file via a Multipart POST using a repeatable file store stream and echo it.")
  public void testRepeatableFileStoreStreamSmallFile() throws Exception {
    File csvInput = new File(getResourceAsUrl(CSV_INPUT_FILE, StreamingTransformationsTestCase.class).toURI());
    HttpPost request = buildMultipartRequest("testRepeatableFileStoreStreamSmallFile", CSV_CONTENT, csvInput);
    HttpResponse response = client.execute(request);

    assertThat(response.getStatusLine().getStatusCode(), is(OK.getStatusCode()));

    String content = IOUtils.toString(response.getEntity().getContent());
    String fileContent = getFileContent(CSV_INPUT_FILE);
    assertThat(content, containsString(fileContent));
  }

  @Test
  @Description("Receive a CSV file via a Multipart POST using a non-repeatable stream and verify the payload is returned untouched.")
  public void testNonRepeatableStreamSmallFile() throws Exception {
    File csvInput = new File(getResourceAsUrl(CSV_INPUT_FILE, StreamingTransformationsTestCase.class).toURI());
    HttpPost request = buildMultipartRequest("testNonRepeatableStreamSmallFile", CSV_CONTENT, csvInput);
    HttpResponse response = client.execute(request);

    assertThat(response.getStatusLine().getStatusCode(), is(OK.getStatusCode()));

    String content = IOUtils.toString(response.getEntity().getContent());
    String fileContent = getFileContent(CSV_INPUT_FILE);
    assertThat(content, containsString(fileContent));
  }

  @Test
  // @FlakyTest()
  @Description("Receive a CSV file via a Multipart POST using a non repeatable stream and consume the file before responding.")
  @Ignore("MULE-18941")
  public void testNonRepeatableStreamSmallFileConsumed() throws Exception {
    File csvInput = new File(getResourceAsUrl(CSV_INPUT_FILE, StreamingTransformationsTestCase.class).toURI());
    HttpPost request = buildMultipartRequest("testNonRepeatableStreamSmallFileConsumed", CSV_CONTENT, csvInput);
    HttpResponse response = client.execute(request);

    assertThat(response.getStatusLine().getStatusCode(), is(OK.getStatusCode()));

    String content = IOUtils.toString(response.getEntity().getContent());

    assertThat(content, is(""));
  }

  @Test
  @Description("Receive a JSON payload via a Repeatable File Store stream and transform it to CSV and back, returning the same payload")
  public void testDWTransformationSmallPayload() throws Exception {
    HttpPost request = buildStringRequest("testDWTransformationSmallPayload", JSON_CONTENT, JSON_INPUT_FILE);
    HttpResponse response = client.execute(request);

    assertThat(response.getStatusLine().getStatusCode(), is(OK.getStatusCode()));

    String content = IOUtils.toString(response.getEntity().getContent());
    String expected = getFileContent(JSON_RESULTS_FILE);
    assertThat(content, is(expected));
  }

  @Test
  @Description("Receive a JSON payload via a Repeatable File Store stream and do two simultaneous transformations.")
  public void testAsyncDWTransformationSmallPayload() throws Exception {
    File output1 = Paths.get(temporaryFolder.getRoot().getAbsolutePath(), "output1.json").toFile();
    File output2 = Paths.get(temporaryFolder.getRoot().getAbsolutePath(), "output2.json").toFile();
    assertThat(output1.exists(), is(false));
    assertThat(output2.exists(), is(false));

    HttpPost request = buildStringRequest("testAsyncDWTransformationSmallPayload", JSON_CONTENT, JSON_INPUT_FILE);
    HttpResponse response = client.execute(request);

    new PollingProber(TIMEOUT_MILLIS, POLL_DELAY_MILLIS).check(new JUnitLambdaProbe(
                                                                                    () -> output1.exists() && output2.exists()));

    assertThat(output1.exists(), is(true));
    assertThat(output2.exists(), is(true));

    assertThat(response.getStatusLine().getStatusCode(), is(OK.getStatusCode()));

    String output1Content = readFileToString(output1);
    String output2Content = readFileToString(output2);
    String expected = getFileContent(JSON_RESULTS_FILE);
    assertThat(output1Content, is(expected));
    assertThat(output2Content, is(expected));

    if (output1.exists()) {
      output1.delete();
    }
    if (output2.exists()) {
      output2.delete();
    }
  }

  @Test
  @Description("Receive a JSON payload via a Repeatable File Store stream and scatter two transformations in different flows, then gather everything into a single collection.")
  public void testScatterGatherDWTransformationSmallPayload() throws Exception {
    HttpPost request = buildStringRequest("testScatterGatherDWTransformationSmallPayload", JSON_CONTENT, JSON_INPUT_FILE);
    HttpResponse response = client.execute(request);

    assertThat(response.getStatusLine().getStatusCode(), is(OK.getStatusCode()));

    String content = IOUtils.toString(response.getEntity().getContent());
    String expected = getFileContent(JSON_INPUT_FILE);
    assertThat(content, is(expected));
  }

  private String getBaseUrl() {
    return "http://localhost:" + httpPort.getValue() + "/";
  }

  private HttpPost buildMultipartRequest(String uri, ContentType mediaType, File file) {
    HttpPost post = new HttpPost(getBaseUrl() + uri);
    post.addHeader("Content-Type", mediaType.getMimeType());
    HttpEntity entity = MultipartEntityBuilder
        .create()
        .addBinaryBody("file", file, mediaType, file.getName())
        .build();
    post.setEntity(entity);
    return post;
  }

  private String getFileContent(String path) throws Exception {
    try (InputStream in = StreamingTransformationsTestCase.class.getClassLoader().getResourceAsStream(path)) {
      assertThat("test file not found: " + path, in, is(notNullValue()));
      return IOUtils.toString(in);
    }
  }

  private HttpPost buildStringRequest(String uri, ContentType mediaType, String path) throws Exception {
    String content = getFileContent(path);
    HttpPost post = new HttpPost(getBaseUrl() + uri);
    post.addHeader("Content-Type", mediaType.getMimeType());
    HttpEntity entity = new StringEntity(content);
    post.setEntity(entity);
    return post;
  }
}
