/*
 * (c) 2003-2023 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.test.module.xa.jms;

import static com.mulesoft.mule.test.module.xa.jms.JmsMessageStorage.cleanUpQueue;
import static java.lang.System.currentTimeMillis;
import static java.util.Collections.emptyMap;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.equalTo;
import static org.junit.Assert.fail;
import static org.mule.extensions.jms.api.exception.JmsError.TIMEOUT;
import static org.mule.functional.junit4.matchers.MessageMatchers.hasPayload;
import static org.mule.tck.junit4.matcher.ErrorTypeMatcher.errorType;

import org.mule.functional.api.exception.ExpectedError;
import org.mule.functional.api.flow.FlowRunner;
import org.mule.functional.junit4.MuleArtifactFunctionalTestCase;
import org.mule.runtime.api.el.BindingContext;
import org.mule.runtime.api.message.Error;
import org.mule.runtime.api.message.Message;
import org.mule.runtime.api.metadata.DataType;
import org.mule.runtime.api.metadata.MediaType;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.core.api.el.ExpressionManager;
import org.mule.runtime.core.privileged.exception.EventProcessingException;
import org.mule.tck.junit4.rule.SystemProperty;
import org.mule.test.runner.ArtifactClassLoaderRunnerConfig;

import java.util.Map;
import java.util.Optional;

import javax.inject.Inject;

import io.qameta.allure.Step;
import org.junit.Before;
import org.junit.Rule;

public abstract class JmsAbstractTestCase extends MuleArtifactFunctionalTestCase {

  protected static final String DESTINATION_VAR = "destination";
  protected static final String MAXIMUM_WAIT_VAR = "maximumWait";

  protected static final String PUBLISHER_FLOW = "publisher";
  protected static final String CONSUMER_FLOW = "consumer";

  protected static final int TIMEOUT_MILLIS = 10000;
  protected static final int POLL_DELAY_MILLIS = 100;
  protected static final String MAX_REDELIVERY = "max.redelivery";
  protected String destination;
  protected long maximumWait = 10000;

  @Inject
  public ExpressionManager expressionManager;

  @Rule
  public ExpectedError expectedError = ExpectedError.none();

  @Rule
  public SystemProperty maxRedelivery = new SystemProperty(MAX_REDELIVERY, "0");

  @Override
  protected void doSetUpBeforeMuleContextCreation() throws Exception {
    super.doSetUpBeforeMuleContextCreation();
  }

  @Before
  public void setUp() {
    cleanUpQueue();
  }

  @Override
  protected void doTearDown() throws Exception {
    cleanUpQueue();
  }

  protected String newDestination(String name) {
    return name + currentTimeMillis();
  }

  protected void publish(Object message, MediaType mediaType) throws Exception {
    publish(message, destination, mediaType);
  }

  protected void publish(Object message, String destination, MediaType mediaType) throws Exception {
    publish(message, destination, emptyMap(), mediaType);
  }

  @Step("Run publish flow message to destination: {destination} with media type and flow vars")
  protected void publish(Object message, String destination, Map<String, Object> flowVars, MediaType mediaType) throws Exception {
    FlowRunner publisher = flowRunner(PUBLISHER_FLOW)
        .withPayload(message)
        .withMediaType(mediaType)
        .withVariable(DESTINATION_VAR, destination);
    flowVars.forEach(publisher::withVariable);
    publisher.run();
  }

  protected Message consume(String destination) throws Exception {
    return consume(destination, emptyMap(), maximumWait);
  }

  protected Message consume(String destination, long maximumWait) throws Exception {
    return consume(destination, emptyMap(), maximumWait);
  }

  @Step("Run consume message flow from dest: {destination} with flow vars and maximum wait")
  protected Message consume(String destination, Map<String, Object> flowVars, long maximumWait) throws Exception {
    FlowRunner consumer = flowRunner(CONSUMER_FLOW)
        .withVariable(DESTINATION_VAR, destination)
        .withVariable(MAXIMUM_WAIT_VAR, maximumWait);
    flowVars.forEach(consumer::withVariable);
    return consumer.run().getMessage();
  }

  @Step("Check for no messages on dest: {destination}")
  protected void assertEmptyDestination(String destination) throws Exception {
    boolean emptyDestination = false;
    try {
      consume(destination, 2000);
    } catch (EventProcessingException exception) {
      Optional<Error> error = exception.getEvent().getError();
      if (error.isPresent() && errorType(TIMEOUT).matches(error.get().getErrorType())) {
        emptyDestination = true;
      }
    } catch (Throwable throwable) {
      fail("Unexpected exception was caught when trying to find out if destination was empty");
    }

    if (!emptyDestination) {
      fail("Destination is not empty");
    }
  }

  @Step("Check for message on dest: {destination}")
  protected void assertMessageOnDestination(String message, String destination) throws Exception {
    assertThat(consume(destination), hasPayload(equalTo(message)));
  }

  public Object evaluate(String expression, Object object) {
    TypedValue typedValue;

    if (object instanceof TypedValue) {
      typedValue = (TypedValue) object;
    } else {
      typedValue = new TypedValue<>(object, DataType.OBJECT);
    }

    return expressionManager.evaluate(expression, BindingContext.builder().addBinding("payload", typedValue).build())
        .getValue();
  }
}
