/*
 * (c) 2003-2023 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.test.transaction;

import org.junit.Rule;
import org.junit.runners.Parameterized;
import org.mule.functional.junit4.MuleArtifactFunctionalTestCase;
import org.mule.runtime.api.component.AbstractComponent;
import org.mule.runtime.api.exception.MuleException;
import org.mule.runtime.api.util.concurrent.Latch;
import org.mule.runtime.core.api.event.CoreEvent;
import org.mule.runtime.core.api.processor.Processor;
import org.mule.runtime.core.api.processor.strategy.ProcessingStrategyFactory;
import org.mule.runtime.core.api.util.func.CheckedRunnable;
import org.mule.tck.junit4.rule.SystemProperty;
import org.mule.test.runner.RunnerDelegateTo;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;

import static java.lang.System.clearProperty;
import static java.lang.System.setProperty;
import static java.lang.Thread.currentThread;
import static java.util.Arrays.asList;
import static org.hamcrest.CoreMatchers.containsString;
import static org.hamcrest.CoreMatchers.is;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.collection.IsCollectionWithSize.hasSize;
import static org.hamcrest.collection.IsIterableContainingInOrder.contains;
import static org.hamcrest.core.Every.everyItem;
import static org.junit.Assert.fail;
import static org.mule.runtime.core.api.transaction.TransactionCoordination.isTransactionActive;

@RunnerDelegateTo(Parameterized.class)
public abstract class AbstractTransactionWithRoutersTestCase extends MuleArtifactFunctionalTestCase {

  protected static final String TX_MESSAGE = "Kangaroo";
  protected static final String OTHER_TX_MESSAGE = "Uruguayan";
  protected static final String CPU_LIGHT = "cpuLight";
  protected static final String IO = "io";
  protected static final String UBER = "uber";
  protected static final String DEFAULT_PROCESSING_STRATEGY_CLASSNAME =
      "org.mule.runtime.core.internal.processor.strategy.TransactionAwareStreamEmitterProcessingStrategyFactory";
  protected static final String PROACTOR_PROCESSING_STRATEGY_CLASSNAME =
      "org.mule.runtime.core.internal.processor.strategy.TransactionAwareProactorStreamEmitterProcessingStrategyFactory";

  protected static List<Thread> threads;
  protected static List<String> payloads;
  protected static List<Boolean> runsInTx;
  protected static Latch latch;

  @Parameterized.Parameters
  public static List<String> parameters() {
    return asList(DEFAULT_PROCESSING_STRATEGY_CLASSNAME, PROACTOR_PROCESSING_STRATEGY_CLASSNAME);
  }

  @Rule
  public SystemProperty message = new SystemProperty("firstValue", TX_MESSAGE);

  @Rule
  public SystemProperty otherMessage = new SystemProperty("otherValue", OTHER_TX_MESSAGE);

  private final String processingStrategyFactoryClassname;

  public AbstractTransactionWithRoutersTestCase(String processingStrategyFactoryClassname) {
    this.processingStrategyFactoryClassname = processingStrategyFactoryClassname;
  }

  @Override
  protected void doSetUpBeforeMuleContextCreation() throws Exception {
    super.doSetUpBeforeMuleContextCreation();
    setDefaultProcessingStrategyFactory(processingStrategyFactoryClassname);
  }

  @Override
  protected void doSetUp() throws Exception {
    super.doSetUp();
    threads = new ArrayList<>();
    payloads = new ArrayList<>();
    latch = new Latch();
    runsInTx = new CopyOnWriteArrayList<>();
  }

  @Override
  protected void doTearDownAfterMuleContextDispose() throws Exception {
    super.doTearDownAfterMuleContextDispose();
    clearDefaultProcessingStrategyFactory();
  }

  protected abstract String getTransactionConfigFile();

  @Override
  protected String[] getConfigFiles() {
    String configFile = getTransactionConfigFile();
    if (PROACTOR_PROCESSING_STRATEGY_CLASSNAME.equals(processingStrategyFactoryClassname)) {
      return new String[] {configFile, "scheduler-custom-dedicated-pool-config.xml"};
    } else {
      return new String[] {configFile};
    }
  }


  protected void onProcessingStrategy(CheckedRunnable withEmmiter, CheckedRunnable withProactor) {
    if (DEFAULT_PROCESSING_STRATEGY_CLASSNAME.equals(processingStrategyFactoryClassname)) {
      withEmmiter.run();
    } else if (PROACTOR_PROCESSING_STRATEGY_CLASSNAME.equals(processingStrategyFactoryClassname)) {
      withProactor.run();
    } else {
      fail("Unknown processing strategy " + processingStrategyFactoryClassname);
    }
  }

  protected void runsInSameTransaction(String flowName, String... expectedPayloads) throws Exception {
    runsInSameTransaction(flowName, false, expectedPayloads);
  }

  protected void runsInSameTransactionAsync(String flowName, String... expectedPayloads) throws Exception {
    runsInSameTransaction(flowName, true, expectedPayloads);
  }

  protected void runsInSameTransaction(String flowName, boolean async, String... expectedPayloads) throws Exception {
    // Checks that all the points captured by the ThreadCaptor are processing within the Transaction, the threads
    // are (as should be) the same, and the payloads are the expected ones
    flowRunner(flowName).run();
    if (async) {
      latch.await();
    }
    assertThat(threads, hasSize(expectedPayloads.length));
    assertThat(threads, everyItem(is(threads.get(0))));
    assertThat(runsInTx, everyItem(is(true)));
    // Since there is no concurrency, the element must match in exact order
    assertThat(payloads, contains(expectedPayloads));
  }

  protected void runsInSameTransactionWithErrors(String flow) throws Exception {
    runsInSameTransaction(flow, TX_MESSAGE, OTHER_TX_MESSAGE, "Error with " + TX_MESSAGE, OTHER_TX_MESSAGE);
  }

  protected void assertThreadType(Thread thread, String type) {
    assertThat(thread.getName(), containsString(type + "."));
  }

  protected void setDefaultProcessingStrategyFactory(String classname) {
    setProperty(ProcessingStrategyFactory.class.getName(), classname);
  }

  protected void clearDefaultProcessingStrategyFactory() {
    clearProperty(ProcessingStrategyFactory.class.getName());
  }

  public static class ThreadCaptor extends AbstractComponent implements Processor {

    @Override
    public CoreEvent process(CoreEvent event) throws MuleException {
      threads.add(currentThread());
      payloads.add(event.getMessage().getPayload().getValue().toString());
      runsInTx.add(isTransactionActive());
      return event;
    }
  }

  public static class ThreadCaptorAsync extends AbstractComponent implements Processor {

    @Override
    public CoreEvent process(CoreEvent event) throws MuleException {
      runsInTx.add(isTransactionActive());
      latch.release();
      return event;
    }
  }
}
