/*
 * (c) 2003-2019 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.test.transaction.xa.bti;

import static java.util.Collections.emptyList;
import static java.util.Collections.singletonMap;
import static org.hamcrest.Matchers.hasSize;
import static org.junit.Assert.assertThat;

import org.mule.extension.db.integration.model.DerbyTestDatabase;

import com.mulesoft.mule.test.transaction.xa.AbstractXaTransactionalTestCase;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.junit.Test;
import org.junit.runners.Parameterized;

public class DbBtiXaTransactionalTestCase extends AbstractXaTransactionalTestCase {

  private static DbConfig config1 = new DbConfig("dbConfig", singletonMap("db", "target/muleEmbeddedDB"));
  private static DbConfig config2 = new DbConfig("dbConfig", singletonMap("db", "target/muleEmbeddedDB2"));
  private static List<DbConfig> derbyConfigs = new ArrayList<>();

  static {
    derbyConfigs.add(config1);
    derbyConfigs.add(config2);
  }

  @Override
  protected String getTransactionManagerResource() {
    return "bti-config.xml";
  }

  @Parameterized.Parameters(name = "{2}")
  public static List<Object[]> parameters() {
    DerbyTestDatabase derbyTestDatabase = new DerbyTestDatabase();
    ArrayList<Object[]> objects = new ArrayList<>();
    objects
        .add(new Object[] {"derby-xa-db-config.xml", derbyTestDatabase, derbyTestDatabase.getDbType(), emptyList(), derbyConfigs,
        });
    return objects;
  }

  @Parameterized.Parameter(4)
  public List<DbConfig> configs;

  @Override
  protected String[] getFlowConfigurationResources() {
    return new String[] {getTransactionManagerResource(), "xa-transactional-config.xml"};
  }

  @Test
  public void commitTxUpdateOutsideABlock() throws Exception {
    executeTransaction("commitTxUpdateOutsideABlock");
    validateDbState(MERCURY);
  }

  @Test
  public void commitAnOptionalTxUpdateIsANonTxBlock() throws Exception {
    executeTransaction("commitAnOptionalTxUpdateIsANonTxBlock");
    validateDbState(MERCURY);
  }

  @Test
  public void commitNonTxUpdateInATxBlock() throws Exception {
    executeTransaction("commitNonTxUpdateInATxBlock");
    validateDbState(MERCURY);
  }

  @Test
  public void rollbackTxUpdateInATxBlock() throws Exception {
    executeTransaction("rollbackTxUpdateInATxBlock");
    validateDbState(MARS);
  }

  @Test
  public void rollbackOptionalTxUpdateInATxBlock() throws Exception {
    executeTransaction("rollbackOptionalTxUpdateInATxBlock");
    validateDbState(MARS);
  }

  @Test
  public void twoUpdatesInTheSameTxBlockAreCommited() throws Exception {
    executeTransaction("twoUpdatesInTheSameTxBlockAreCommited");
    validateDbState(MERCURY);
  }

  @Test
  public void consecutiveTxBlocksUsingDynamicConfigs() throws Exception {
    executeTransaction("consecutiveTxBlocksUsingDynamicConfigs");
    for (DbConfig config : configs) {
      validateDbState(MERCURY, config.getName(), config.getVariables());
    }
  }

  @Test
  public void twoUpdatesInTheSameTxBlockFromDifferentDynamicConfigsWithRollback() throws Exception {
    executeTransaction("twoUpdatesInTheSameTxBlockFromDifferentDynamicConfigsWithRollback");
    for (DbConfig config : configs) {
      validateDbState(MARS, config.getName(), config.getVariables());
    }
  }

  @Test
  public void consumingStreamDoesntCloseConnectionInTx() throws Exception {
    List<Map> value =
        (List<Map>) executeSucessfullTransaction("consumingStreamDoesntCloseConnectionInTx").getMessage().getPayload().getValue();
    assertThat(value, hasSize(6));
  }

  @Override
  protected Map<String, Object> additionalVariables() {
    HashMap<String, Object> stringObjectHashMap = new HashMap<>();
    stringObjectHashMap.put("db", config1.getVariables().get("db"));
    stringObjectHashMap.put("db2", config2.getVariables().get("db"));
    return stringObjectHashMap;
  }

  @Override
  public List<DbConfig> additionalConfigs() {
    return configs;
  }
}
