/*
 * (c) 2003-2019 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.test.transaction.xa.jms;

import static com.mulesoft.mule.test.transaction.xa.jms.TransactionalAction.COMMIT;
import static com.mulesoft.mule.test.transaction.xa.jms.TransactionalAction.ROLLBACK;
import static java.lang.System.currentTimeMillis;
import static org.hamcrest.CoreMatchers.is;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.mule.tck.probe.PollingProber.probe;

import org.mule.functional.api.flow.FlowRunner;
import org.mule.functional.junit4.MuleArtifactFunctionalTestCase;
import org.mule.runtime.api.message.Message;
import org.mule.runtime.core.api.construct.Flow;
import org.mule.runtime.core.api.util.UUID;
import org.mule.tck.junit4.matcher.ErrorTypeMatcher;
import org.mule.tck.junit4.rule.SystemProperty;
import org.mule.tck.junit4.rule.SystemPropertyLambda;

import com.mulesoft.mule.test.transaction.xa.XaTestCaseRunnerConfig;

import javax.inject.Inject;

import org.junit.After;
import org.junit.Rule;
import org.junit.Test;

public class JmsXATXTestCase extends MuleArtifactFunctionalTestCase implements XaTestCaseRunnerConfig {

  protected String newDestination(String name) {
    return name + currentTimeMillis();
  }

  @Rule
  public SystemProperty xaListenerDestination =
      new SystemPropertyLambda("jms.xa.destination", () -> newDestination("xa-listener"));

  @Inject
  Flow xaListener;

  @Inject
  Flow xaListenerLimitedRedelivery;

  @Override
  protected String[] getConfigFiles() {
    return new String[] {"jms/jms.xml", "jms/jms-xa.xml", "jms/activemq-xa-jms-config.xml", "jms/activemq-jms-config.xml",
        "bti-config.xml"};
  }

  @After
  public void cleanUpQueues() {
    JmsMessageStorage.cleanUpQueue();
  }

  @Test
  public void publishXa() throws Exception {
    String publishXa = newDestination("publishXa");
    String message = createMessage();
    publishXa(message, publishXa);
    Message consume = consume(publishXa);
    Object value = consume.getPayload().getValue();
    assertThat(value, is(message));
  }

  @Test
  public void consumeXa() throws Exception {
    String consumeXa = newDestination("consumeXa");
    String message = createMessage();
    publish(message, consumeXa);
    Message consume = consumeXa(consumeXa);

    Object value = consume.getPayload().getValue();
    assertThat(value, is(message));
  }

  @Test
  public void rollbackOnPublishedMessage() throws Exception {
    String publishXa = newDestination("rollbackOnPublishedMessage");
    String message = createMessage();
    publishXa(message, publishXa, ROLLBACK);

    consumeAndTimeout(publishXa, 2000);
  }

  @Test
  public void rollbackOnConsumedMessage() throws Exception {
    String consumeXa = newDestination("rollbackOnConsumedMessage");
    String message = createMessage();
    publish(message, consumeXa);
    consumeXa(consumeXa, ROLLBACK);
    Message consume = consume(consumeXa);
    Object value = consume.getPayload().getValue();
    assertThat(value, is(message));
  }

  @Test
  public void consumeAndPublish() throws Exception {
    String consumeDestination = newDestination("consumeDestination");
    String publishDestination = newDestination("publishDestination");

    String message = createMessage();
    publish(message, consumeDestination);
    consumePublishXa(consumeDestination, publishDestination, COMMIT);

    Message consume = consume(publishDestination);
    Object value = consume.getPayload().getValue();
    assertThat(value, is(message));
  }

  @Test
  public void consumeAndPublish2Configs() throws Exception {
    String consumeDestination = newDestination("consumeDestination");
    String publishDestination = newDestination("publishDestination");

    String message = createMessage();
    publish(message, consumeDestination);
    consumePublishXa2Configs(consumeDestination, publishDestination, COMMIT);

    Message consume = consume(publishDestination);
    Object value = consume.getPayload().getValue();
    assertThat(value, is(message));
  }

  @Test
  public void consumeAndPublishRollback() throws Exception {
    String consumeDestination = newDestination("consumeDestination");
    String publishDestination = newDestination("publishDestination");

    String message = createMessage();
    publish(message, consumeDestination);
    consumePublishXa(consumeDestination, publishDestination, ROLLBACK);

    consumeAndTimeout(publishDestination, 2000);
    Message consume = consume(consumeDestination);
    Object value = consume.getPayload().getValue();
    assertThat(value, is(message));
  }

  @Test
  public void consumeAndPublishRollback2Configs() throws Exception {
    String consumeDestination = newDestination("consumeDestination");
    String publishDestination = newDestination("publishDestination");

    String message = createMessage();
    publish(message, consumeDestination);
    consumePublishXa(consumeDestination, publishDestination, ROLLBACK);

    consumeAndTimeout(publishDestination, 2000);
    Message consume = consume(consumeDestination);
    Object value = consume.getPayload().getValue();
    assertThat(value, is(message));
  }

  @Test
  public void listenerXa() throws Exception {
    xaListener.start();
    ListenerMessage message = new ListenerMessage(createMessage(), COMMIT.name(), "NONE");
    publish(message, xaListenerDestination.getValue());
    publish(message, xaListenerDestination.getValue());
    publish(message, xaListenerDestination.getValue());
    publish(message, xaListenerDestination.getValue());
    publish(message, xaListenerDestination.getValue());
    probe(20000, 100, () -> 5 == JmsMessageStorage.receivedMessages());
    ListenerMessage value = (ListenerMessage) JmsMessageStorage.pollMuleMessage().getPayload().getValue();
    assertThat(message.getMessage(), is(value.getMessage()));
    xaListener.stop();
    consumeAndTimeout(xaListenerDestination.getValue(), 2000);
  }

  @Test
  public void listenerXaRollback() throws Exception {
    xaListenerLimitedRedelivery.start();
    ListenerMessage message = new ListenerMessage(createMessage(), ROLLBACK.name(), "NONE");
    publish(message, xaListenerDestination.getValue());
    probe(20000, 100, () -> 5 == JmsMessageStorage.receivedMessages());
  }

  @Test
  public void listenerXaAndPublishMessage() throws Exception {
    xaListener.start();

    ListenerMessage message = new ListenerMessage(createMessage(), COMMIT.name(), "PUBLISH");
    String publishDestination = newDestination("listenerXaAndPublishMessage");
    message.setDestination(publishDestination);

    publish(message, xaListenerDestination.getValue());

    ListenerMessage value = (ListenerMessage) JmsMessageStorage.pollMuleMessage().getPayload().getValue();
    assertThat(message.getMessage(), is(value.getMessage()));
    consumeAndTimeout(xaListenerDestination.getValue(), 2000);

    ListenerMessage publishedMessage = (ListenerMessage) consume(publishDestination).getPayload().getValue();
    assertThat(publishedMessage.getMessage(), is(value.getMessage()));
  }


  private void consumeAndTimeout(String destination, int maximumWait) throws Exception {
    flowRunner("consume")
        .withVariable("destination", destination)
        .withVariable("maximumWait", maximumWait)
        .runExpectingException(ErrorTypeMatcher.errorType("JMS", "TIMEOUT"));
  }

  public void publishXa(Object payload, String destination) throws Exception {
    publishXa(payload, destination, TransactionalAction.COMMIT);
  }

  public void publishXa(Object payload, String destination, TransactionalAction action) throws Exception {
    FlowRunner flowRunner = flowRunner("publishXa")
        .withPayload(payload)
        .withVariable("destination", destination)
        .withVariable("txAction", action.name());
    executeTransactionalFlowrunner(flowRunner, action);
  }

  private Object executeTransactionalFlowrunner(FlowRunner flowRunner, TransactionalAction action) throws Exception {
    if (action.equals(ROLLBACK)) {
      Exception exception = flowRunner.runExpectingException();
      if (exception.getMessage().equals("TX.")) {
        return exception;
      } else {
        throw exception;
      }
    } else {
      return flowRunner.run().getMessage();
    }
  }

  public void publish(Object payload, String destination) throws Exception {
    flowRunner("publish")
        .withPayload(payload)
        .withVariable("destination", destination)
        .run();
  }

  public Message consumeXa(String destination) throws Exception {
    return (Message) consumeXa(destination, TransactionalAction.COMMIT);
  }

  public Object consumeXa(String destination, TransactionalAction action) throws Exception {
    FlowRunner flowRunner = flowRunner("consumeXa")
        .withVariable("destination", destination)
        .withVariable("txAction", action.name());

    return executeTransactionalFlowrunner(flowRunner, action);
  }

  public Message consume(String destination) throws Exception {
    return flowRunner("consume")
        .withVariable("destination", destination)
        .run().getMessage();
  }

  public Message consume(String destination, long maximumWait) throws Exception {
    return flowRunner("consume")
        .withVariable("destination", destination)
        .withVariable("maximumWait", maximumWait)
        .run().getMessage();
  }

  public void consumePublishXa(String consumeDestination, String publishDestination, TransactionalAction action)
      throws Exception {
    executeTransactionalFlowrunner(flowRunner("consume-publishXa")
        .withVariable("consumeDestination", consumeDestination)
        .withVariable("publishDestination", publishDestination)
        .withVariable("txAction", action.name()), action);
  }

  public void consumePublishXa2Configs(String consumeDestination, String publishDestination, TransactionalAction action)
      throws Exception {
    executeTransactionalFlowrunner(flowRunner("consume-publish-2-configs-Xa")
        .withVariable("consumeDestination", consumeDestination)
        .withVariable("publishDestination", publishDestination)
        .withVariable("txAction", action.name()), action);
  }

  private String createMessage() {
    return UUID.getUUID();
  }
}
