/**
 *
 * (c) 2013 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.datamapper.function;

import com.mulesoft.datamapper.transform.function.Calendar2DateFunction;
import com.mulesoft.datamapper.transform.function.Calendar2LongFunction;
import com.mulesoft.datamapper.transform.function.Calendar2NumFunction;
import com.mulesoft.datamapper.transform.function.Calendar2StrFunction;
import com.mulesoft.datamapper.transform.function.Date2CalendarFunction;
import com.mulesoft.datamapper.transform.function.Long2CalendarFunction;
import com.mulesoft.datamapper.transform.function.Str2CalendarFunction;
import org.apache.commons.lang.time.DateUtils;
import org.jetel.ctl.Stack;
import org.jetel.ctl.extensions.TLFunctionAnnotation;
import org.jetel.ctl.extensions.TLFunctionCallContext;
import org.jetel.ctl.extensions.TLFunctionLibrary;
import org.jetel.ctl.extensions.TLFunctionPrototype;

import java.util.Date;
import java.util.TimeZone;

public class CalendarLib extends TLFunctionLibrary
{
    @Override
    public String getName()
    {
        return "Calendar";
    }

    @Override
    public TLFunctionPrototype getExecutable(String functionName)
    {
        if (functionName.equals("calendar2date"))
        {
            return new Calendar2DateCTLFunction();
        }
        else if (functionName.equals("calendar2long"))
        {
            return new Calendar2LongCTLFunction();
        }
        else if (functionName.equals("calendar2str"))
        {
            return new Calendar2StrCTLFunction();
        }
        else if (functionName.equals("calendar2number"))
        {
            return new Calendar2NumCTLFunction();
        }
        else if (functionName.equals("long2calendar"))
        {
            return new Long2CalendarCTLFunction();
        }
        else if (functionName.equals("date2calendar"))
        {
            return new Date2CalendarCTLFunction();
        }
        else if (functionName.equals("str2calendar"))
        {
            return new Str2CalendarCTLFunction();
        }

        throw new IllegalArgumentException("Unknown function '" + functionName + "'");
    }


    @TLFunctionAnnotation("Transforms a calendar to a date.")
    public static Date calendar2date(TLFunctionCallContext context, String calendar)
    {
        return Calendar2DateFunction.calendarToDate(calendar);
    }

    @TLFunctionAnnotation()
    public static Long calendar2long(TLFunctionCallContext context, String calendar)
    {
        return Calendar2LongFunction.calendarToLong(calendar);
    }

    @TLFunctionAnnotation()
    public static String calendar2str(TLFunctionCallContext context, String calendar, String pattern)
    {
        return Calendar2StrFunction.calendarToStr(calendar, pattern);
    }

    @TLFunctionAnnotation()
    public static Integer calendar2number(TLFunctionCallContext context, String calendar, Integer unit)
    {
        return Calendar2NumFunction.calendarToNumber(calendar, unit);
    }

    @TLFunctionAnnotation()
    public static String long2calendar(TLFunctionCallContext context, Long milliseconds)
    {
        return Long2CalendarFunction.longToCalendar(milliseconds, DateUtils.UTC_TIME_ZONE);
    }

    @TLFunctionAnnotation()
    public static String long2calendar(TLFunctionCallContext context, Long milliseconds, String timeZone)
    {
        return Long2CalendarFunction.longToCalendar(milliseconds, TimeZone.getTimeZone(timeZone));
    }


    @TLFunctionAnnotation()
    public static String date2calendar(TLFunctionCallContext context, Date date)
    {
        return Date2CalendarFunction.dateToCalendar(date, DateUtils.UTC_TIME_ZONE);
    }

    @TLFunctionAnnotation()
    public static String date2calendar(TLFunctionCallContext context, Date date, String timeZone)
    {
        return Date2CalendarFunction.dateToCalendar(date, TimeZone.getTimeZone(timeZone));
    }

    @TLFunctionAnnotation()
    public static String str2calendar(TLFunctionCallContext context, String calendarString, String pattern)
    {
        return Str2CalendarFunction.stringToCalendar(calendarString, pattern, DateUtils.UTC_TIME_ZONE);
    }

    @TLFunctionAnnotation()
    public static String str2calendar(TLFunctionCallContext context, String calendarString, String pattern, String timeZone)
    {
        return Str2CalendarFunction.stringToCalendar(calendarString, pattern, TimeZone.getTimeZone(timeZone));
    }


    public class Calendar2DateCTLFunction implements TLFunctionPrototype
    {
        @Override
        public void init(TLFunctionCallContext context)
        {

        }

        @Override
        public void execute(Stack stack, TLFunctionCallContext context)
        {
            stack.push(calendar2date(context, stack.popString()));
        }
    }

    public class Calendar2LongCTLFunction implements TLFunctionPrototype
    {
        @Override
        public void init(TLFunctionCallContext context)
        {

        }

        @Override
        public void execute(Stack stack, TLFunctionCallContext context)
        {
            stack.push(calendar2long(context, stack.popString()));
        }
    }

    public class Calendar2NumCTLFunction implements TLFunctionPrototype
    {
        @Override
        public void init(TLFunctionCallContext context)
        {

        }

        @Override
        public void execute(Stack stack, TLFunctionCallContext context)
        {
            stack.push(calendar2number(context, stack.popString(), stack.popInt()));
        }
    }

    public class Calendar2StrCTLFunction implements TLFunctionPrototype
    {
        @Override
        public void init(TLFunctionCallContext context)
        {

        }

        @Override
        public void execute(Stack stack, TLFunctionCallContext context)
        {
            stack.push(calendar2str(context, stack.popString(), stack.popString()));
        }
    }

    public class Long2CalendarCTLFunction implements TLFunctionPrototype
    {
        @Override
        public void init(TLFunctionCallContext context)
        {

        }

        @Override
        public void execute(Stack stack, TLFunctionCallContext context)
        {
            if (context.getParams().length == 1)
            {
                stack.push(long2calendar(context, stack.popLong()));
            }
            else
            {
                stack.push(long2calendar(context, stack.popLong(), stack.popString()));
            }
        }
    }

    public class Date2CalendarCTLFunction implements TLFunctionPrototype
    {
        @Override
        public void init(TLFunctionCallContext context)
        {

        }

        @Override
        public void execute(Stack stack, TLFunctionCallContext context)
        {
            if (context.getParams().length == 1)
            {
                stack.push(date2calendar(context, stack.popDate()));
            }
            else
            {
                stack.push(date2calendar(context, stack.popDate(), stack.popString()));
            }
        }
    }

    public class Str2CalendarCTLFunction implements TLFunctionPrototype
    {
        @Override
        public void init(TLFunctionCallContext context)
        {

        }

        @Override
        public void execute(Stack stack, TLFunctionCallContext context)
        {
            if (context.getParams().length == 2)
            {
                stack.push(str2calendar(context, stack.popString(), stack.popString()));
            }
            else
            {
                stack.push(str2calendar(context, stack.popString(), stack.popString(), stack.popString()));
            }
        }
    }
}
