/**
 *
 * (c) 2012 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.datamapper.lookup;

import org.jetel.data.DataRecord;
import org.jetel.data.Defaults;
import org.jetel.data.HashKey;
import org.jetel.data.RecordKey;
import org.jetel.data.lookup.Lookup;
import org.jetel.data.lookup.LookupTable;
import org.jetel.exception.*;
import org.jetel.graph.GraphElement;
import org.jetel.graph.TransformationGraph;
import org.jetel.metadata.DataRecordMetadata;
import org.jetel.util.primitive.TypedProperties;
import org.jetel.util.property.ComponentXMLAttributes;
import org.jetel.util.string.StringUtils;

import org.w3c.dom.Element;

import java.util.Iterator;

public class MuleFlowLookupTable extends GraphElement implements LookupTable {

    private static final String XML_LOOKUP_KEY = "key";
    private static final String XML_FLOW_REF = "flow-ref";
    private static final String XML_LOOKUP_TYPE_FLOW_REF_LOOKUP = "muleFlowLookup";

    private final static String[] REQUESTED_ATTRIBUTE = {XML_ID_ATTRIBUTE, XML_TYPE_ATTRIBUTE, XML_METADATA_ID, XML_LOOKUP_KEY, XML_FLOW_REF};

    private String metadataName;
    private DataRecordMetadata metadata;
    private String[] keys;
    private RecordKey indexKey;
    private String flowRefName;


    public MuleFlowLookupTable(String id, String metadataName, String[] keys, String flowRefName) {
        super(id);
        this.flowRefName = flowRefName;
        this.setMetadataName(metadataName);
        this.setKeys(keys);

    }

    public boolean isPutSupported() {
        return false;
    }

    public boolean isRemoveSupported() {
        return false;
    }

    public DataRecordMetadata getMetadata() {
        return metadata;
    }

    public DataRecordMetadata getKeyMetadata() throws ComponentNotReadyException {
        if (!isInitialized()) {
            throw new NotInitializedException(this);
        }
        return getIndexKey().generateKeyRecordMetadata();
    }

    public boolean put(DataRecord dataFields) {
        return false;
    }

    public boolean remove(DataRecord dataFields) {
        return false;
    }

    public boolean remove(HashKey hashKey) {
        return false;
    }

    public Lookup createLookup(RecordKey recordKey) throws ComponentNotReadyException {
        return new MuleFlowLookup(this, recordKey, MuleFlowRefAdaptorFactory.createMuleFlowAdaptor(getKeys(), getFlowRefName(), getGraph().getDictionary()));
    }

    public Lookup createLookup(RecordKey recordKey, DataRecord dataFields) throws ComponentNotReadyException {
        return new MuleFlowLookup(this, recordKey, MuleFlowRefAdaptorFactory.createMuleFlowAdaptor(getKeys(), getFlowRefName(), getGraph().getDictionary()));
    }

    @Override
    public void setCurrentPhase(int phase) {

    }

    public Iterator<DataRecord> iterator() {
        return null;
    }

    public synchronized void init() throws ComponentNotReadyException {
        if (isInitialized()) {
            // throw new
            // IllegalStateException("The lookup table has already been initialized!");
            return;
        }
        super.init();

        if (getMetadata() == null) {
            setMetadata(getGraph().getDataRecordMetadata(getMetadataName(), true));
        }
        if (getMetadata() == null) {
            throw new ComponentNotReadyException("Metadata " + StringUtils.quote(getMetadataName()) + " does not exist!!!");
        }

        if (getIndexKey() == null) {
            setIndexKey(new RecordKey(getKeys(), getMetadata()));
        }
        getIndexKey().init();
        getIndexKey().setEqualNULLs(true);

    }


    @Override
    public ConfigurationStatus checkConfig(ConfigurationStatus status) {
        super.checkConfig(status);

        if (getMetadata() == null) {
            setMetadata(getGraph().getDataRecordMetadata(getMetadataName(), false));
        }

        if (getIndexKey() == null) {
            setIndexKey(new RecordKey(getKeys(), getMetadata()));
        }

        try {
            getIndexKey().init();
        } catch (NullPointerException e) {
            status.add(new ConfigurationProblem("Key metadata are null.", ConfigurationStatus.Severity.WARNING, this,
                    ConfigurationStatus.Priority.NORMAL, XML_LOOKUP_KEY));
            setIndexKey(null); // we have to create it once again in init method
            // after
            // creating metadata from stub
        } catch (RuntimeException e) {
            status.add(new ConfigurationProblem(e.getMessage(), ConfigurationStatus.Severity.ERROR, this,
                    ConfigurationStatus.Priority.NORMAL, XML_LOOKUP_KEY));
        }

        return status;
    }

    public static MuleFlowLookupTable fromProperties(TypedProperties properties)
            throws AttributeNotFoundException, GraphConfigurationException {

        for (String property : REQUESTED_ATTRIBUTE) {
            if (!properties.containsKey(property)) {
                throw new AttributeNotFoundException(property);
            }
        }
        String type = properties.getStringProperty(XML_TYPE_ATTRIBUTE);
        if (!type.equalsIgnoreCase("muleFlowLookup")) {
            throw new GraphConfigurationException("Can't create simple lookup table from type " + type);
        }

        String[] keys = properties.getStringProperty(XML_LOOKUP_KEY).split(Defaults.Component.KEY_FIELDS_DELIMITER_REGEX);
        String metadata = properties.getStringProperty(XML_METADATA_ID);

        // set any property from the xml

        final String flowName = properties.getStringProperty(XML_FLOW_REF);


        MuleFlowLookupTable lookupTable = new MuleFlowLookupTable(properties.getStringProperty(XML_ID_ATTRIBUTE), metadata, keys, flowName);

        if (properties.containsKey(XML_NAME_ATTRIBUTE)) {
            lookupTable.setName(properties.getStringProperty(XML_NAME_ATTRIBUTE));
        }


        return lookupTable;
    }

    public static MuleFlowLookupTable fromXML(TransformationGraph graph, Element nodeXML) throws XMLConfigurationException {
        ComponentXMLAttributes xattribs = new ComponentXMLAttributes(nodeXML, graph);
        MuleFlowLookupTable lookupTable = null;
        String id;
        String type;

        // reading obligatory attributes
        try {
            id = xattribs.getString(XML_ID_ATTRIBUTE);
            type = xattribs.getString(XML_TYPE_ATTRIBUTE);
        } catch (AttributeNotFoundException ex) {
            throw new XMLConfigurationException("Can't create lookup table - " + ex.getMessage(), ex);
        }

        // check type
        if (!type.equalsIgnoreCase(XML_LOOKUP_TYPE_FLOW_REF_LOOKUP)) {
            throw new XMLConfigurationException("Can't create simple lookup table from type " + type);
        }

        // create simple lookup table
        try {
            String[] keys = xattribs.getString(XML_LOOKUP_KEY).split(Defaults.Component.KEY_FIELDS_DELIMITER_REGEX);
            // set any property from the xml
            final String flowName = xattribs.getString(XML_FLOW_REF);
            //

            String metadata = xattribs.getString(XML_METADATA_ID);
            lookupTable = new MuleFlowLookupTable(id, metadata, keys, flowName);
            lookupTable.setGraph(graph);
            if (xattribs.exists(XML_NAME_ATTRIBUTE)) {
                lookupTable.setName(xattribs.getString(XML_NAME_ATTRIBUTE));
            }

            return lookupTable;

        } catch (Exception ex) {
            throw new XMLConfigurationException("can't create simple lookup table", ex);
        }
    }

    public String getMetadataName() {
        return metadataName;
    }

    public void setMetadataName(String metadataName) {
        this.metadataName = metadataName;
    }

    public void setMetadata(DataRecordMetadata metadata) {
        this.metadata = metadata;
    }

    public String[] getKeys() {
        return keys;
    }

    public void setKeys(String[] keys) {
        this.keys = keys;
    }

    public RecordKey getIndexKey() {
        return indexKey;
    }

    public void setIndexKey(RecordKey indexKey) {
        this.indexKey = indexKey;
    }


    public String getFlowRefName() {
        return flowRefName;
    }

    public void setFlowRefName(String flowRefName) {
        this.flowRefName = flowRefName;
    }
}
