/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.module.boot;

import com.mulesource.licm.EnterpriseLicenseKey;
import com.mulesource.licm.LicenseKeyException;
import com.mulesource.licm.LicenseManagementFactory;
import com.mulesource.licm.LicenseManager;

import java.io.File;
import java.io.IOException;
import java.net.URL;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class LicenseKeyHandler
{
    private static final Logger logger = LoggerFactory.getLogger(LicenseKeyHandler.class);

    private LicenseManager licenseManager;

    public File LICENSE_LOCATION;

    public static final String LICENSE_FILE_NAME = "muleLicenseKey.lic";

    // key only available if it has been validated
    private static EnterpriseLicenseKey key = null;

    public LicenseKeyHandler()
    {
        try
        {
            licenseManager = LicenseManagementFactory.getInstance().createLicenseManager(
                LicenseManagementFactory.MULE_EE);

            // LICENSE_LOCATION = new File("");
        }
        catch (LicenseKeyException e)
        {
            logger.error("Error creating License Manager!", e);
            throw new IllegalStateException();
        }

    }

    public EnterpriseLicenseKey install(EnterpriseLicenseKey key) throws LicenseKeyException
    {
        try
        {
            // incase
            licenseManager.unInstall();
            licenseManager.install(key);
            EnterpriseLicenseKey installedKey = licenseManager.validate(LicenseManagementFactory.MULE_EE);
            LICENSE_LOCATION = lookupMuleBaseConf();
            moveLicenseKeyFileToMuleBaseConf();
            File licenseKeyFile = new File(key.getLicenseKeyFile());
            licenseKeyFile.delete();
            return installedKey;
        }
        catch (LicenseKeyException e)
        {
            logger.error("Couldn't install license key!", e);
            throw e;
        }

    }

    public void unInstall() throws LicenseKeyException
    {
        try
        {
            licenseManager.unInstall();
        }
        catch (LicenseKeyException e)
        {
            logger.error("Couldn't uninstall license key!", e);
            throw e;
        }

    }

    public EnterpriseLicenseKey validate() throws LicenseKeyException
    {
        EnterpriseLicenseKey key = null;

        try
        {
            key = licenseManager.validate(LicenseManagementFactory.MULE_EE);
            // this key was just generated, others will be where they should
            if (key.isEvaluation())
            {
                LICENSE_LOCATION = lookupMuleBaseConf();
                moveLicenseKeyFileToMuleBaseConf();
            }
        }
        catch (LicenseKeyException e)
        {
            System.out.println("Couldn't validate license key. Please contact MuleSource Support, support@mulesoft.com, to renew your license.");
        }
        // set the key
        LicenseKeyHandler.key = key;

        return key;

    }

    /**
     * Move the license key file back to $MULE_HOME/conf from the working directory
     * 
     * @return
     */
    private void moveLicenseKeyFileToMuleBaseConf()
    {

        URL licenseUrl = Thread.currentThread().getContextClassLoader().getResource(LICENSE_FILE_NAME);

        if (licenseUrl == null)
        {
            File file = new File(LICENSE_FILE_NAME);
            if (file.exists())
            {
                File target = new File(LICENSE_LOCATION, LICENSE_FILE_NAME);
                boolean success = file.renameTo(target);
                if (!success)
                    try
                    {
                        MuleBootstrapUtils.safeCopyFile(file, target);
                        file.delete();
                    }
                    catch (Exception e)
                    {
                        logger.debug("Couldn't move license file from: " + file.getAbsolutePath() + "to: "
                                     + target.getAbsolutePath());
                    }

            }
        }
    }

    private File lookupMuleBaseConf()
    {
        File muleHome = null;
        String muleHomeVar = System.getProperty("mule.home");

        if (muleHomeVar != null && !muleHomeVar.trim().equals("") && !muleHomeVar.equals("%MULE_HOME%"))
        {
            try
            {
                muleHome = new File(muleHomeVar).getCanonicalFile();
            }
            catch (IOException e)
            {
                throw new IllegalStateException("Problem with mule.home!");
            }
        }

        if (muleHome == null || !muleHome.exists() || !muleHome.isDirectory())
        {
            throw new IllegalArgumentException(
                "Either MULE_HOME is not set or does not contain a valid directory.");
        }

        File muleBase = null;
        File muleBaseConf = null;

        String muleBaseVar = System.getProperty("mule.base");

        if (muleBaseVar != null && !muleBaseVar.trim().equals("") && !muleBaseVar.equals("%MULE_BASE%"))
        {
            try
            {
                muleBase = new File(muleBaseVar).getCanonicalFile();

            }
            catch (IOException e)
            {
                throw new IllegalStateException("Problem with mule.home!");
            }
        }

        if (muleBase == null)
        {
            muleBase = muleHome;
        }

        muleBaseConf = new File(muleBase, "conf");

        return muleBaseConf;
    }

    /**
     * @return the key
     */
    public static EnterpriseLicenseKey getKey()
    {
        return key;
    }

}
