/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.module.boot;

import org.mule.module.boot.extra.BootstrapModulesLoader;
import org.mule.util.ClassUtils;
import org.mule.util.PropertiesUtils;

import com.mulesource.licm.EnterpriseLicenseKey;
import com.mulesource.licm.LicenseManagementFactory;

import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.util.Date;
import java.util.Properties;

import org.apache.commons.cli.BasicParser;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;
import org.tanukisoftware.wrapper.WrapperListener;
import org.tanukisoftware.wrapper.WrapperManager;
import org.tanukisoftware.wrapper.WrapperSimpleApp;

/**
 * Determine which is the main class to run and delegate control to the Java Service
 * Wrapper.  If OSGi is not being used to boot with, configure the classpath based on
 * the libraries in $MULE_HOME/lib/*
 * <p/>
 * Note: this class is intentionally kept free of any external library dependencies and
 * therefore repeats a few utility methods.
 */
public class MuleBootstrap
{
    private static final String MULE_MODULE_BOOT_POM_FILE_PATH = "META-INF/maven/com.mulesoft.muleesb.modules/mule-module-boot/pom.properties";

    private static final String TESTING_MODE_PROPERTY_NAME = "mule.testingMode";

    public static final String MAIN_CLASS_MULE_SERVER = "org.mule.module.boot.MuleServerWrapper";

    public static final String CLI_OPTIONS[][] = {
        {"main", "true", "Main Class"},
        {"nogui", "false", "Suppress graphical console"},
        {"version", "false", "Show product and version information"},
        {"installLicense", "true", "Replace current license with new license supplied"},
        {"unInstallLicense", "false", "Uninstall current license"},
        {"verifyLicense", "false", "Prints currently installed license details"}
    };
    
    public static void main(String[] args) throws Exception
    {
        // Parse any command line options based on the list above.
        CommandLine commandLine = parseCommandLine(args);
        // Any unrecognized arguments get passed through to the next class (e.g., to Knopflerfish).
        String[] remainingArgs = commandLine.getArgs();

        String mainClassName = commandLine.getOptionValue("main");
        
        EnterpriseLicenseKey key = handleLicenseKey(commandLine);
        
        if (commandLine.hasOption("version"))
        {
            startAndShowVersion(remainingArgs, key);
        }
        else if (mainClassName == null || mainClassName.equals(MAIN_CLASS_MULE_SERVER))
        {
            prepareBootstrapPhase(key);
            System.out.println("Starting the Mule Server...");
            WrapperManager.start((WrapperListener) Class.forName(MAIN_CLASS_MULE_SERVER).newInstance(), remainingArgs);
        }
        else
        {
            // Add the main class name as the first argument to the Wrapper.
            String[] appArgs = new String[remainingArgs.length + 1];
            appArgs[0] = mainClassName;
            System.arraycopy(remainingArgs, 0, appArgs, 1, remainingArgs.length);
            prepareBootstrapPhase(key);
            System.out.println("Starting class " + mainClassName + "...");
            WrapperSimpleApp.main(appArgs);
        }
    }

    private static void startAndShowVersion(String[] remainingArgs, EnterpriseLicenseKey key)
        throws Exception
    {
        prepareBootstrapPhase(key);
        WrapperManager.start(new VersionWrapper(), remainingArgs);
    }
    
    private static void prepareBootstrapPhase(EnterpriseLicenseKey key) throws Exception
    {
        File muleHome = lookupMuleHome();
        File muleBase = lookupMuleBase();

        if (muleBase == null)
        {
            muleBase = muleHome;
        }

        /*
         * CRITICAL: load and configure extra features in classpath *before* 
         * loading other modules, otherwise patch overriding in PTS module
         *  may not work as expected.
         */
        BootstrapModulesLoader.loadModules(MuleBootstrapUtils.getMuleModuleDir(), key);
        
        MuleBootstrapUtils.addLocalJarFilesToClasspath(muleHome, muleBase);
        setSystemMuleVersion();
    }
    
    protected static File lookupMuleHome() throws Exception
    {
        File muleHome = null;
        String muleHomeVar = System.getProperty("mule.home");
        
        if (muleHomeVar != null && !muleHomeVar.trim().equals("") && !muleHomeVar.equals("%MULE_HOME%"))
        {
            muleHome = new File(muleHomeVar).getCanonicalFile();
        }

        if (muleHome == null || !muleHome.exists() || !muleHome.isDirectory())
        {
            throw new IllegalArgumentException("Either MULE_HOME is not set or does not contain a valid directory.");
        }
        return muleHome;
    }
    
    protected static File lookupMuleBase() throws Exception
    {
        File muleBase = null;
        String muleBaseVar = System.getProperty("mule.base");
        
        if (muleBaseVar != null && !muleBaseVar.trim().equals("") && !muleBaseVar.equals("%MULE_BASE%"))
        {
            muleBase = new File(muleBaseVar).getCanonicalFile();
        }
        return muleBase;
    }    
        
    private static void setSystemMuleVersion()
    {
        try
        {
            URL mavenPropertiesUrl = ClassUtils.getResource(MULE_MODULE_BOOT_POM_FILE_PATH, MuleServerWrapper.class);
            Properties mavenProperties = PropertiesUtils.loadProperties(mavenPropertiesUrl);
            
            System.setProperty("mule.version", mavenProperties.getProperty("version"));
            System.setProperty("mule.reference.version", mavenProperties.getProperty("version") + '-' + (new Date()).getTime());
        }
        catch (Exception ignore)
        {
            // ignore;
        }
    }

    /**
     * Parse any command line arguments using the Commons CLI library.
     */
    private static CommandLine parseCommandLine(String[] args) throws ParseException
    {
        Options options = new Options();
        for (int i = 0; i < CLI_OPTIONS.length; i++)
        {
            options.addOption(CLI_OPTIONS[i][0], "true".equalsIgnoreCase(CLI_OPTIONS[i][1]), CLI_OPTIONS[i][2]);
        }
        return new BasicParser().parse(options, args, true);
    }
    
    private static EnterpriseLicenseKey handleLicenseKey(CommandLine commandLine) throws Exception
    {
        LicenseKeyHandler licenseKeyHandler = new LicenseKeyHandler();

        EnterpriseLicenseKey key = LicenseManagementFactory.getInstance().createLicenseKey(
            LicenseManagementFactory.MULE_EE);

        if (commandLine.hasOption("installLicense"))
        {
            String keyRequestFile = commandLine.getOptionValue("installLicense");

            File keyRequest = new File(keyRequestFile);

            if (keyRequest.exists() == false)
            {
                throw new IOException("License key file - " + keyRequestFile + " not found!");
            }

            key.setLicenseKeyFile(keyRequestFile);
            EnterpriseLicenseKey newKey = licenseKeyHandler.install(key);
            System.out.println();
            System.out.println("Installed license key.");
            System.out.println(newKey);
            System.out.println();
            startAndShowVersion(new String[0], key);
            WrapperManager.stop(0);
        }
        else if (commandLine.hasOption("unInstallLicense"))
        {
            licenseKeyHandler.unInstall();
            System.out.println();
            System.out.println("Uninstalled license key.");
            System.out.println();
            startAndShowVersion(new String[0], key);
            WrapperManager.stop(0);
        }
        else if (commandLine.hasOption("verifyLicense"))
        {
            key = licenseKeyHandler.validate();
            if (key != null)
            {
                System.out.println();
                System.out.println("Valid license key --> " + key);
                System.out.println();
            }
            startAndShowVersion(new String[0], key);
            WrapperManager.stop(0);
        }
        else
        {
            key = licenseKeyHandler.validate();
            if (key != null)
            {
                if (System.getProperty(TESTING_MODE_PROPERTY_NAME) != null)
                {
                    System.out.println();
                    System.out.println("Running Mule on testing mode");
                    System.out.println();
                }
                else
                {
                    System.out.println();
                    System.out.println("Valid license key --> " + key);
                    System.out.println();
                }
            }
            else
            {
                startAndShowVersion(new String[0], key);
                WrapperManager.stop(0);
            }

        }
        return key;
    }
}
