/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.module.boot.extra;

import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.prefs.Preferences;

/**
 * Reads configuration of bootstrap modules from conf/extra-bootstrap-modules.xml file
 */
public class BootstrapConfigReader
{
    private static final String CONFIG_FILE = "extra-bootstrap-modules.xml";

    private static List bootstrapModules = new ArrayList();

    /**
     * Reads extra-bootstrap-modules.xml file and populates {@link #bootstrapModules} list
     */
    public static void read()
    {
        // only load this if there is no PreferencesFactory set
        if (System.getProperty("java.util.prefs.PreferencesFactory") == null)
            System.setProperty("java.util.prefs.PreferencesFactory",
                "org.mule.module.boot.prefs.MemoryPreferencesFactory");

        try
        {
            InputStream is = Thread.currentThread().getContextClassLoader().getResourceAsStream(CONFIG_FILE);
            try
            {
                Preferences.importPreferences(is);
                is.close();
            }
            catch (IOException e)
            {
                System.out.println(CONFIG_FILE + " not found. Extra features disabled.");
                bootstrapModules = Collections.EMPTY_LIST;
                return;
            }

            String[] children = Preferences.systemRoot().childrenNames();

            // we should clear this if there is a new read call made
            if (children.length > 0) bootstrapModules.clear();
            for (int i = 0; i < children.length; i++)
            {
                // check for prefs which are not ours
                if (Preferences.systemRoot()
                    .node(children[i])
                    .get("bootstrapLoader", "")
                    .compareToIgnoreCase("") == 0) continue;
                bootstrapModules.add(new BootstrapModuleDescriptor(Preferences.systemRoot().node(children[i])));
            }
        }
        catch (Exception e)
        {
            System.out.println("Invalid " + CONFIG_FILE);
            e.printStackTrace();
            bootstrapModules = Collections.EMPTY_LIST;
        }
    }

    /**
     * @return list of {@link org.mule.modules.boot.BootstrapModuleDescriptor}
     *         instances
     */
    public static List getBootstrapModules()
    {
        return bootstrapModules;
    }
}
