/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.module.reboot;

import static org.tanukisoftware.wrapper.WrapperManager.WRAPPER_LOG_LEVEL_ERROR;
import static org.tanukisoftware.wrapper.WrapperManager.addWrapperEventListener;
import static org.tanukisoftware.wrapper.WrapperManager.log;
import static org.tanukisoftware.wrapper.WrapperManager.start;
import static org.tanukisoftware.wrapper.WrapperManager.stop;
import static org.tanukisoftware.wrapper.event.WrapperEventListener.EVENT_FLAG_LOGGING;

import org.mule.module.boot.LicenseKeyHandler;

import com.mulesource.licm.EnterpriseLicenseKey;
import com.mulesource.licm.LicenseManagementFactory;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.Date;
import java.util.Properties;

import org.apache.commons.cli.BasicParser;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;
import org.tanukisoftware.wrapper.WrapperListener;
import org.tanukisoftware.wrapper.event.WrapperEvent;
import org.tanukisoftware.wrapper.event.WrapperEventListener;

/**
 * This is a copy of the CE version of MuleContainerBootstrap which adds license checking. Make
 * sure that all changes to the CE version of this class are reflected here.
 */
public class MuleContainerBootstrap
{
    private static final String MULE_MODULE_BOOT_POM_FILE_PATH = "META-INF/maven/org.mule.module/mule-module-boot/pom.properties";

    private static final String TESTING_MODE_PROPERTY_NAME = "mule.testingMode";

    public static final String CLI_OPTIONS[][] = {
        {"main", "true", "Main Class"},
        {"installLicense", "true", "Replace current license with new license supplied"},
        {"unInstallLicense", "false", "Uninstall current license"},
        {"verifyLicense", "false", "Prints currently installed license details"},
        {"version", "false", "Show product and version information"}
    };

    public static void main(String[] args) throws Exception
    {
        CommandLine commandLine;
        // Any unrecognized arguments get passed through to the next class
        String[] remainingArgs = null;

        Integer requestedExitCode = null;
        String errorMessage = null;
        try
        {
            // Parse any command line options based on the list above.
            commandLine = parseCommandLine(args);
            // Any unrecognized arguments get passed through to the next class
            remainingArgs = commandLine.getArgs();

            requestedExitCode = handleLicenseKey(commandLine);
        }
        catch (Exception ex)
        {
            System.out.println(ex.getMessage());
            requestedExitCode = 1;
            errorMessage = ex.getMessage();
        }

        if (requestedExitCode != null)
        {
            final String errorMessageToLog = errorMessage;
            addWrapperEventListener(new WrapperEventListener()
            {

                private boolean errorAlreadyLogged = false;

                @Override
                public void fired(WrapperEvent event)
                {
                    // Checks if there is an errorMessage involved in the exit request
                    if (!errorAlreadyLogged && errorMessageToLog != null)
                    {
                        log(WRAPPER_LOG_LEVEL_ERROR, errorMessageToLog);
                        errorAlreadyLogged = true;
                    }

                }

            }, EVENT_FLAG_LOGGING);

            start(new WrapperListener()
            {
                @Override
                public Integer start(String[] args)
                {
                    return null;
                }

                @Override
                public int stop(int exitCode)
                {
                    return exitCode;
                }

                @Override
                public void controlEvent(int event)
                {

                }
            }, new String[0]);
            stop(requestedExitCode);
        }
        else
        {
            prepareBootstrapPhase();

            System.out.println("Starting the Mule Container...");
            start(new EEMuleContainerWrapper(), remainingArgs);
        }
    }

    private static void prepareBootstrapPhase() throws Exception
    {
        File muleHome = lookupMuleHome();
        File muleBase = lookupMuleBase();
        if (muleBase == null)
        {
            muleBase = muleHome;
        }

        setSystemMuleVersion();
    }

    public static File lookupMuleHome() throws Exception
    {
        File muleHome = null;
        String muleHomeVar = System.getProperty("mule.home");

        if (muleHomeVar != null && !muleHomeVar.trim().equals("") && !muleHomeVar.equals("%MULE_HOME%"))
        {
            muleHome = new File(muleHomeVar).getCanonicalFile();
        }

        if (muleHome == null || !muleHome.exists() || !muleHome.isDirectory())
        {
            throw new IllegalArgumentException("Either the system property mule.home is not set or does not contain a valid directory.");
        }
        return muleHome;
    }

    public static File lookupMuleBase() throws Exception
    {
        File muleBase = null;
        String muleBaseVar = System.getProperty("mule.base");

        if (muleBaseVar != null && !muleBaseVar.trim().equals("") && !muleBaseVar.equals("%MULE_BASE%"))
        {
            muleBase = new File(muleBaseVar).getCanonicalFile();
        }
        return muleBase;
    }

    private static void setSystemMuleVersion()
    {
        InputStream propertiesStream = null;
        try
        {
            URL mavenPropertiesUrl = MuleContainerBootstrapUtils.getResource(MULE_MODULE_BOOT_POM_FILE_PATH, MuleContainerWrapper.class);
            propertiesStream = mavenPropertiesUrl.openStream();

            Properties mavenProperties = new Properties();
            mavenProperties.load(propertiesStream);

            System.setProperty("mule.version", mavenProperties.getProperty("version"));
            System.setProperty("mule.reference.version", mavenProperties.getProperty("version") + '-' + (new Date()).getTime());
        }
        catch (Exception ignore)
        {
            // ignore;
        }
        finally
        {
            if (propertiesStream != null)
            {
                try
                {
                    propertiesStream.close();
                }
                catch (IOException iox)
                {
                    // ignore
                }
            }
        }
    }

    /**
     * Parse any command line arguments using the Commons CLI library.
     */
    private static CommandLine parseCommandLine(String[] args) throws ParseException
    {
        Options options = new Options();
        for (int i = 0; i < CLI_OPTIONS.length; i++)
        {
            options.addOption(CLI_OPTIONS[i][0], "true".equalsIgnoreCase(CLI_OPTIONS[i][1]), CLI_OPTIONS[i][2]);
        }
        return new BasicParser().parse(options, args, true);
    }

    private static Integer handleLicenseKey(CommandLine commandLine) throws Exception
    {
        LicenseKeyHandler licenseKeyHandler = new LicenseKeyHandler();

        EnterpriseLicenseKey key = LicenseManagementFactory.getInstance().createLicenseKey(
            LicenseManagementFactory.MULE_EE);

        if (commandLine.hasOption("installLicense"))
        {
            String keyRequestFile = commandLine.getOptionValue("installLicense");

            File keyRequest = new File(keyRequestFile);

            if (keyRequest.exists() == false)
            {
                throw new IOException("License key file - " + keyRequestFile + " not found!");
            }

            key.setLicenseKeyFile(keyRequestFile);
            EnterpriseLicenseKey newKey = licenseKeyHandler.install(key);
            System.out.println();
            System.out.println("Installed license key.");
            System.out.println(newKey);
            System.out.println();
            return 0;
        }
        else if (commandLine.hasOption("unInstallLicense"))
        {
            licenseKeyHandler.unInstall();
            System.out.println();
            System.out.println("Uninstalled license key.");
            System.out.println();
            return 0;
        }
        else if (commandLine.hasOption("verifyLicense"))
        {
            key = licenseKeyHandler.validate();
            if (key != null)
            {
                System.out.println();
                System.out.println("Valid license key --> " + key);
                System.out.println();
                return 0;
            }
            return 1;
        }
        else
        {
            key = licenseKeyHandler.validate();
            if (key != null)
            {
                // TODO: Improve this
                if (System.getProperty(TESTING_MODE_PROPERTY_NAME) != null)
                {
                    EEMuleContainerBootstrap.setKey(key, true);
                    System.out.println();
                    System.out.println("Running Mule on testing mode");
                    System.out.println();
                }
                else
                {
                    EEMuleContainerBootstrap.setKey(key, false);
                    System.out.println();
                    System.out.println("Valid license key --> " + key);
                    System.out.println();
                }
            }
            else
            {
                System.out.println("invalid license key, stopping now");
                return 1;
            }
        }
        return null;
    }
}

