/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.module.reboot;

import java.io.File;
import java.io.FileFilter;
import java.io.FileInputStream;
import java.io.IOException;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.jar.JarEntry;
import java.util.jar.JarInputStream;
import java.util.regex.Pattern;

public class PatchScanner
{
    protected static Pattern PATCH_PATTERN = Pattern.compile(".*[\\\\/]patch-\\w+-\\d{8}.jar$");
    protected File patchDirectory;
    protected Map<String, File> classes;
    protected Map<String, Set<File>> duplicates;

    public PatchScanner(File patchDirectory)
    {
        super();
        this.patchDirectory = patchDirectory;
        this.classes = new HashMap<String, File>();
        this.duplicates = new HashMap<String, Set<File>>();
    }

    public List<File> listPatchJars()
    {
        List<File> patchJars = getPatchJars();
        scanForDuplicateClasses(patchJars);

        if (duplicates.size() > 0)
        {
            createErrorMessageAndThrow();
        }

        return patchJars;
    }

    protected List<File> getPatchJars()
    {
        File[] jars = patchDirectory.listFiles(new FileFilter()
        {
            public boolean accept(File pathname)
            {
                try
                {
                    return PATCH_PATTERN.matcher(pathname.getCanonicalPath()).matches();
                }
                catch(IOException e)
                {
                    throw new RuntimeException(e.getMessage());
                }
            }
        });

        if (jars != null)
        {
            return Arrays.asList(jars);
        }
        return Collections.emptyList();
    }

    private void scanForDuplicateClasses(List<File> patchJars)
    {
        try
        {
            for (File jar : patchJars)
            {
                scanJarForDuplicateClasses(jar);
            }
        }
        catch (IOException iox)
        {
            throw new IllegalStateException(iox);
        }
    }

    protected void scanJarForDuplicateClasses(File jarFile) throws IOException
    {
        JarInputStream jarStream = null;
        try
        {
            jarStream = new JarInputStream(new FileInputStream(jarFile));
            scanStreamForDuplicateClasses(jarStream, jarFile);
        }
        finally
        {
            // cannot use IOUtils here, the classpath is not fully set up yet
            if (jarStream != null)
            {
                jarStream.close();
            }
        }
    }

    private void scanStreamForDuplicateClasses(JarInputStream jarStream, File sourceFile) throws IOException
    {
        JarEntry entry;
        while ((entry = jarStream.getNextJarEntry()) != null)
        {
            if (entry.isDirectory())
            {
                continue;
            }

            String name = entry.getName();
            if (name.startsWith("META-INF"))
            {
                continue;
            }

            File patchFile = classes.get(name);
            if (patchFile != null)
            {
                addDuplicate(name, patchFile, sourceFile);
            }

            classes.put(name, sourceFile);
        }
    }

    private void addDuplicate(String name, File fileOne, File fileTwo)
    {
        Set<File> set = duplicates.get(name);
        if (set == null)
        {
            set = new HashSet<File>();
            duplicates.put(name, set);
        }

        set.add(fileOne);
        set.add(fileTwo);
    }

    private void createErrorMessageAndThrow()
    {
        StringBuilder errorMessage = new StringBuilder(512);
        errorMessage.append("\n\nDuplicate classes in patch jars detected:\n\n");

        for (Map.Entry<String, Set<File>> entry : duplicates.entrySet())
        {
            String className = entry.getKey();
            Set<File> containingJars = entry.getValue();

            errorMessage.append("Patch jars ");
            errorMessage.append(containingJars.toString());
            errorMessage.append(" all contain duplicate class ");

            className = className.replace('/', '.');
            className = className.replace(".class", "");
            errorMessage.append(className);

            errorMessage.append("\n");
        }

        throw new IllegalStateException(errorMessage.toString());
    }
}
