/*
 * CloverETL Engine - Java based ETL application framework.
 * Copyright (c) Javlin, a.s. (info@cloveretl.com).  Use is subject to license terms.
 *
 * www.cloveretl.com
 */
package com.mulesoft.datamapper.transform.converter;

import java.lang.reflect.Array;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.net.URI;
import java.net.URL;
import java.sql.Time;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.beanutils.ConversionException;
import org.apache.commons.beanutils.Converter;
import org.apache.commons.beanutils.converters.BigIntegerConverter;
import org.apache.commons.beanutils.converters.BooleanConverter;
import org.apache.commons.beanutils.converters.ByteConverter;
import org.apache.commons.beanutils.converters.CalendarConverter;
import org.apache.commons.beanutils.converters.CharacterConverter;
import org.apache.commons.beanutils.converters.DateConverter;
import org.apache.commons.beanutils.converters.DoubleConverter;
import org.apache.commons.beanutils.converters.FloatConverter;
import org.apache.commons.beanutils.converters.IntegerConverter;
import org.apache.commons.beanutils.converters.LongConverter;
import org.apache.commons.beanutils.converters.ShortConverter;
import org.apache.commons.beanutils.converters.SqlDateConverter;
import org.apache.commons.beanutils.converters.SqlTimeConverter;
import org.apache.commons.beanutils.converters.SqlTimestampConverter;
import org.apache.commons.beanutils.converters.StringConverter;
import org.apache.commons.beanutils.converters.URLConverter;
import org.jetel.data.primitive.Decimal;
import org.jetel.util.string.CloverString;

/**
 * Utility to convert various scalar values between types.
 *
 * @author jan.michalica (info@cloveretl.com) (c) Javlin, a.s. (www.cloveretl.com)
 * @created 24.7.2012
 */
public class ScalarValueConversion
{

    private static ScalarValueConversion sharedInstance;

    public static ScalarValueConversion theInstance()
    {

        if (sharedInstance == null)
        {
            sharedInstance = new ScalarValueConversion();
        }
        return sharedInstance;
    }

    private Map<Class, Converter> converterMap = new HashMap<Class, Converter>();
    private Map<Class, Class> boxMap = new HashMap<Class, Class>();

    protected ScalarValueConversion()
    {
        init();
    }

    protected void init()
    {

        register(new BigIntegerConverter(), BigInteger.class);
        register(new BooleanConverter(Boolean.FALSE), Boolean.TYPE);
        register(new BooleanConverter(), Boolean.class);
        register(new CharacterConverter((char) 0), Character.TYPE);
        register(new CharacterConverter(), Character.class);
        register(new ByteConverter((byte) 0), Byte.TYPE);
        register(new ByteConverter(), Byte.class);
        register(new DoubleConverter((double) 0), Double.TYPE);
        register(new DoubleConverter(), Double.class);
        register(new FloatConverter((float) 0), Float.TYPE);
        register(new FloatConverter(), Float.class);
        register(new IntegerConverter(0), Integer.TYPE);
        register(new IntegerConverter(), Integer.class);
        register(new LongConverter((long) 0), Long.TYPE);
        register(new LongConverter(), Long.class);
        register(new ShortConverter((short) 0), Short.TYPE);
        register(new ShortConverter(), Short.class);
        register(new StringConverter(), String.class);
        register(new URLConverter(), URL.class);
        register(new DateConverter(), Date.class);
        register(new CalendarConverter(), Calendar.class);
        register(new DateConverter(), Date.class);
        register(new SqlDateConverter(), java.sql.Date.class);
        register(new SqlTimeConverter(), Time.class);
        register(new SqlTimestampConverter(), Timestamp.class);
        /*
         * custom converters
         */
        register(new UriConverter(), URI.class);
        register(new CloverStringConverter(), CloverString.class);
        register(new EnumConverter(), Enum.class);
        register(new DecimalConverter(), Decimal.class);
        register(new BigDecimalConverter(), BigDecimal.class);

        registerBox(byte.class, Byte.class);
        registerBox(int.class, Integer.class);
        registerBox(long.class, Long.class);
        registerBox(double.class, Double.class);
        registerBox(boolean.class, Boolean.class);
        registerBox(char.class, Character.class);
        registerBox(float.class, Float.class);

    }

    private void register(Converter converter, Class<?> clazz)
    {
        converterMap.put(clazz, converter);
    }

    private void registerBox(Class<?> primitive, Class<?> box)
    {
        boxMap.put(primitive, box);
    }

    public Object convert(Object value, Class<?> desiredType) throws ConversionException
    {

        if (value == null)
        {
            return null;
        }
        if (desiredType == null)
        {
            throw new ConversionException("target type null");
        }

        if (value != null)
        {
            if (desiredType.isAssignableFrom(value.getClass()) || isBoxingOf(desiredType, value.getClass()))
            {
                return value;
            }
        }
        return doConvert(desiredType, value);
    }


    private boolean isBoxingOf(Class<?> primitive, Class<?> boxType)
    {
        if (primitive.isPrimitive())
        {
            return boxMap.get(primitive).equals(boxType);
        }
        else
        {
            return false;
        }
    }

    private <T> Object doConvert(Class<T> desiredType, Object value)
    {
        if (converterMap.containsKey(desiredType))
        {
            return converterMap.get(desiredType).convert(desiredType, value);
        }
        else
        {
            for (Map.Entry<Class, Converter> classConverterEntry : converterMap.entrySet())
            {
                if (desiredType.isAssignableFrom(classConverterEntry.getKey()))
                {
                    return converterMap.get(desiredType).convert(desiredType, value);
                }
            }
        }
        throw new RuntimeException("No converter found for type " + desiredType);
    }

    public List<?> convertList(Object value, Class<?> desiredType)
    {
        if (value == null)
        {
            return null;
        }
        if (value instanceof List)
        {

            List<?> valueAsList = (List<?>) value;
            List<Object> result = new ArrayList<Object>(valueAsList.size());
            for (Object element : valueAsList)
            {
                result.add(convert(element, desiredType));
            }
            return result;
        }
        else if (value.getClass().isArray())
        {
            int length = Array.getLength(value);
            List<Object> result = new ArrayList<Object>(length);
            for (int i = 0; i < length; i++)
            {
                result.add(convert(Array.get(value, i), desiredType));
            }
            return result;
        }
        ArrayList<Object> result = new ArrayList<Object>();
        result.add(convert(value, desiredType));
        return result;

    }

}
