/**
 *
 * (c) 2011 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */

package com.mulesoft.mule.module.datamapper;


import org.mule.api.MuleContext;
import org.mule.api.MuleEvent;
import org.mule.api.context.MuleContextAware;
import org.mule.api.transport.PropertyScope;

import com.mulesoft.mule.module.datamapper.api.GraphExecutor;
import com.mulesoft.mule.module.datamapper.api.GraphExecutorFactory;
import com.mulesoft.mule.module.datamapper.api.OutputArgumentHandler;
import com.mulesoft.mule.module.datamapper.api.TransformationResult;
import com.mulesoft.mule.module.datamapper.api.exception.DataMapperCreationException;
import com.mulesoft.mule.module.datamapper.api.exception.DataMapperExecutionException;
import com.mulesoft.mule.module.datamapper.clover.CloverGraphExecutorFactory;
import com.mulesoft.mule.module.datamapper.clover.impl.CloverEngineManagerContainer;

import java.util.HashMap;
import java.util.Map;

import javax.annotation.PostConstruct;

/**
 * Data Mapper Mule entry point.
 *
 * @author Alberto Pose <alberto.pose@mulesoft.com>
 * @author Ján Michalica <jan.michalica@javlin.eu>
 * @author Lukáš Krejčí <lukas.krejci@javlin.eu>
 */

public class DataMapperModule implements MuleContextAware
{


    public static final String MULE_CONTEXT = "____@muleContext@____";
    public static final String MULE_EVENT = "____@muleEvent@____";

    /**
     * Relative or absolute path to the CloverETL graph to execute.
     */

    private String transformationGraphPath;


    /**
     * Factory class for Graph Executor
     */
    private GraphExecutorFactory executorFactory;
    /**
     * Mule Context
     */
    private MuleContext muleContext;
    private GraphExecutor graphExecutor;


    private int initialPoolSize;
    private String outputContentType;
    private String outputEncoding;


    @PostConstruct
    public void initialise()
    {
        try
        {
            if(executorFactory == null)
            {
                executorFactory = createGraphExecutorFactory();
                executorFactory.init();
                graphExecutor = executorFactory.createGraphExecutor(transformationGraphPath);
            }
        }
        catch (Throwable e)
        {
            throw new RuntimeException(e);
        }
    }

    private GraphExecutorFactory createGraphExecutorFactory()
    {
        return new CloverGraphExecutorFactory(muleContext, CloverEngineManagerContainer.INSTANCE.getCloverEngineManager(), initialPoolSize);

    }


    public void dispose()
    {

        executorFactory.dispose();
    }

    /**
     * Transform an input data format into another. User can specify additional arguments to enrich the result.
     *
     * @param event          The mule event
     * @param input          The input data to be transformed
     * @param inputArguments Input Arguments. Additional arguments to the data mapping
     * @param streaming
     * @param pipeSize
     * @return Headers and input returned form the transformation.
     *         <p/>
     *         {@sample.xml ../../../doc/data-mapper.xml.sample
     *         data-mapper:transform}
     */

    public Object transform(final MuleEvent event, Object input,
                            Map<String, Object> inputArguments, boolean streaming, int pipeSize) throws DataMapperExecutionException, DataMapperCreationException
    {

        Map<String, Object> arguments;
        if (inputArguments != null)
        {
            arguments = new HashMap<String, Object>(inputArguments);
        }
        else
        {
            arguments = new HashMap<String, Object>();
        }
        arguments.put(MULE_CONTEXT, muleContext);
        arguments.put(MULE_EVENT, event);
        if (streaming)
        {
            return graphExecutor.executeStreaming(input, pipeSize, arguments, new OutputArgumentHandler()
            {
                @Override
                public void addArgument(String name, Object value)
                {
                    event.getMessage().setProperty(name, value, PropertyScope.INVOCATION);
                }
            });
        }
        else
        {
            TransformationResult result = graphExecutor.execute(input, arguments);
            Map<String, Object> otherResults = result.getOutputArguments();
            for (Map.Entry<String, Object> outputArgument : otherResults.entrySet())
            {
                if (outputArgument.getValue() != null)
                {
                    event.getMessage().setProperty(outputArgument.getKey(), outputArgument.getValue(), PropertyScope.INVOCATION);
                }
            }

            return result.getPayload();
        }

    }


    public void setTransformationGraphPath(String transformationGraphPath)
    {
        this.transformationGraphPath = transformationGraphPath;
    }

    public void setInitialPoolSize(int initialPoolSize)
    {
        this.initialPoolSize = initialPoolSize;
    }

    @Override
    public void setMuleContext(MuleContext muleContext)
    {
        this.muleContext = muleContext;
    }

    public String getOutputContentType()
    {
        if (outputContentType == null)
        {
            outputContentType = graphExecutor.getOutputContentType();
        }
        return outputContentType;
    }

    public String getOutputEncoding()
    {
        if (outputEncoding == null)
        {
            outputEncoding = graphExecutor.getOutputEncoding();
        }
        return outputEncoding;
    }
}
