/**
 *
 * (c) 2011 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */

package com.mulesoft.mule.module.datamapper.clover.impl;

import com.mulesoft.mule.module.datamapper.api.FutureCallback;
import com.mulesoft.mule.module.datamapper.api.TransformationEngine;
import com.mulesoft.mule.module.datamapper.api.TransformationResult;
import com.mulesoft.mule.module.datamapper.api.exception.DataMapperExecutionException;
import com.mulesoft.mule.module.datamapper.clover.MappingFormat;
import com.mulesoft.mule.module.datamapper.util.Counter;

import java.lang.reflect.Field;
import java.util.Collection;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jetel.exception.ComponentNotReadyException;
import org.jetel.graph.Node;
import org.jetel.graph.Result;
import org.jetel.graph.TransformationGraph;
import org.jetel.graph.dictionary.Dictionary;
import org.jetel.graph.runtime.GraphRuntimeContext;
import org.jetel.graph.runtime.SingleThreadWatchDog;
import org.jetel.graph.runtime.WatchDog;
import org.jetel.util.string.StringUtils;

/**
 * A wrapper for the Clover engine that implements the TransformationEngine Interface
 */
public class CloverEngineImpl implements TransformationEngine<TransformationGraph>
{

    private static Log log = LogFactory.getLog(CloverEngineImpl.class);

    public static final int DEFAULT_MAX_GRAPH = Integer.getInteger("com.mulesoft.datamapper.max_graph", 0);

    public static final int DEFAULT_MAX_NODES = Integer.getInteger("com.mulesoft.datamapper.max_nodes", 0);

    private long runs;

    private DataMapperThreadManager threadManager;

    private Counter graphCounter = new Counter();

    private CloverEngineImpl(DataMapperThreadManager threadManager)
    {
        this.threadManager = threadManager;
    }

    public CloverEngineImpl()
    {
        this(new DataMapperThreadManager(DEFAULT_MAX_GRAPH, DEFAULT_MAX_NODES));
    }

    public TransformationResult execute(TransformationGraph graph, GraphRuntimeContext context, Map<String, Object> inputArgument)
            throws DataMapperExecutionException
    {


        validateGraph(graph);

        Dictionary dictionary = graph.getDictionary();
        try
        {
            for (Map.Entry<String, Object> inArgumentEntry : inputArgument.entrySet())
            {
                dictionary.setValue(inArgumentEntry.getKey(), inArgumentEntry.getValue());
            }

        }
        catch (ComponentNotReadyException e)
        {
            log.error("Exception while generating dictionary", e);
            throw new DataMapperExecutionException(e);
        }

        ClassLoader muleContextClassLoader = Thread.currentThread().getContextClassLoader();
        try
        {
             /*
            * create watch dog for graph run
            */
            final SingleThreadWatchDog watchDog = new SingleThreadWatchDog(graph, context);
            watchDog.init();
            final CloverResult cloverResult = new CloverResult(watchDog.call(), dictionary);
            cloverResult.setCauseException(watchDog.getCauseException());
            return cloverResult;
        }
        catch (Exception e)
        {
            throw new DataMapperExecutionException("Data Mapper execution failed.", e);
        }
        finally
        {
            //Clover changes
            Thread.currentThread().setContextClassLoader(muleContextClassLoader);
        }

    }

    public MappingFormat getOutputType(TransformationGraph graph)
    {
        Map<String, Node> nodes = graph.getNodes();
        Collection<Node> values = nodes.values();
        for (Node value : values)
        {
            String type = value.getType();
            MappingFormat[] mappingFormats = MappingFormat.values();
            for (MappingFormat mappingFormat : mappingFormats)
            {
                if (mappingFormat.getWriterType().equals(type))
                {
                    return mappingFormat;
                }
            }
        }
        return null;
    }


    public String getOutputEncoding(TransformationGraph graph)
    {
        Map<String, Node> nodes = graph.getNodes();
        Collection<Node> values = nodes.values();
        for (Node value : values)
        {
            String type = value.getType();
            MappingFormat[] mappingFormats = MappingFormat.values();
            for (MappingFormat mappingFormat : mappingFormats)
            {
                if (mappingFormat.getWriterType().equals(type))
                {
                    return mappingFormat.getEncoding(value);
                }
            }
        }
        return null;
    }


    @Override
    public TransformationResult execute(TransformationGraph graph, Map<String, Object> inputArgument) throws DataMapperExecutionException
    {
        return execute(graph, graph.getRuntimeContext(), inputArgument);
    }


    public void executeLater(TransformationGraph graph,
                             GraphRuntimeContext context, Map<String, Object> inputArgument, FutureCallback<TransformationResult> callback) throws DataMapperExecutionException
    {

        validateGraph(graph);

        Dictionary dictionary = graph.getDictionary();
        try
        {
            for (Map.Entry<String, Object> inArgumentEntry : inputArgument.entrySet())
            {
                dictionary.setValue(inArgumentEntry.getKey(), inArgumentEntry.getValue());
            }

        }
        catch (ComponentNotReadyException e)
        {
            log.error("Exception while generating dictionary", e);
            throw new DataMapperExecutionException(e);
        }

        /*
        * create watch dog for graph run
        */
        WatchDog watchDog = new WatchDog(graph, context);
        getThreadManager().initWatchDog(watchDog);
        context.setRunId(nextRunId());


        FutureCallbackProxy proxy = new FutureCallbackProxy(graph, callback);
        try
        {
            getThreadManager().executeWatchDog(watchDog, proxy);
            graphCounter.increment();
        }
        catch (RuntimeException e)
        {
            throw new DataMapperExecutionException(e);
        }


    }

    @Override
    public void executeLater(TransformationGraph graph, Map<String, Object> inputArgument, FutureCallback<TransformationResult> callback)
            throws DataMapperExecutionException
    {
        executeLater(graph, graph.getRuntimeContext(), inputArgument, callback);
    }

    private void validateGraph(TransformationGraph graph) throws DataMapperExecutionException
    {
        /*
           * fail if not initialized
           */
        if (!graph.isInitialized())
        {
            throw new DataMapperExecutionException("Graph is not initialized.");
        }
    }

    private class FutureCallbackProxy implements FutureCallback<Result>
    {

        private final FutureCallback<TransformationResult>[] targets;
        private TransformationGraph graph;

        public FutureCallbackProxy(TransformationGraph graph, FutureCallback<TransformationResult>... target)
        {
            this.targets = target;
            this.graph = graph;
        }

        private void release()
        {
            try
            {
                graphCounter.decrement();
            }
            catch (InterruptedException e)
            {
                log.warn("Failed to decrement running graph count.", e);
            }
        }

        @Override
        public void done(Result result)
        {
            release();
            for (FutureCallback<TransformationResult> target : targets)
            {
                target.done(new CloverResult(result, graph.getDictionary()));
            }

        }

        @Override
        public void failed(Throwable throwable)
        {
            release();
            for (FutureCallback<TransformationResult> target : targets)
            {
                target.failed(throwable);
            }

        }

        @Override
        public void cancelled()
        {
            release();
            for (FutureCallback<TransformationResult> target : targets)
            {
                target.cancelled();
            }

        }
    }

    public synchronized void dispose()
    {
        log.info("Stopping clover transformation graph engine.");
        /*
           * prevent execution of more graphs
           */
        getThreadManager().free();
        log.info("Waiting for " + graphCounter.getCount() + " graphs to finish execution.");
        try
        {
            graphCounter.await(0);
        }
        catch (Exception e)
        {
            log.warn("Interruted while waiting for grahs to finish.", e);
        }

        /*
           * abort graphs execution
           */
        log.info("Aborting " + graphCounter.getCount() + " running graphs.");
        getThreadManager().freeNow();

        log.info("Stopped clover transformation graph engine.");
    }

    private synchronized long nextRunId()
    {
        return runs++;
    }

    public DataMapperThreadManager getThreadManager()
    {
        return threadManager;
    }


}
