/**
 *
 * (c) 2011 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */

package com.mulesoft.mule.module.datamapper.clover.impl;

import com.mulesoft.mule.module.datamapper.api.FutureCallback;

import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.FutureTask;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jetel.graph.Result;
import org.jetel.graph.runtime.CloverFuture;
import org.jetel.graph.runtime.IThreadManager;
import org.jetel.graph.runtime.WatchDog;

/**
 * Thread manager based on com.cloveretl.server.graph.runtime.ServerThreadManager
 * with added support for future callback.
 *
 */
public class DataMapperThreadManager implements IThreadManager
{

    private static Log logger = LogFactory.getLog(DataMapperThreadManager.class);

    private ThreadPoolExecutor watchdogExecutor;

    private ThreadPoolExecutor nodeExecutor;

    private int maxNodes = 0;

    /**
     * Counter specifying the number current running nodes.
     */
    private int runningNodes = 0;


    /**
     * Graph executor's constructor.
     *
     * @param maxGraphs maximum number of simultaneously running graphs; zero means without limit
     * @param maxNodes  maximum number of simultaneously running nodes; zero means without limit
     */
    public DataMapperThreadManager(int maxGraphs, int maxNodes)
    {
        this.maxNodes = maxNodes;

        if (maxGraphs <= 0)
        {
            watchdogExecutor = (ThreadPoolExecutor) Executors.newCachedThreadPool(new WatchdogThreadFactory());
        }
        else
        {
            watchdogExecutor = (ThreadPoolExecutor) Executors.newFixedThreadPool(maxGraphs, new WatchdogThreadFactory());
        }
        if (maxNodes <= 0)
        {
            nodeExecutor = (ThreadPoolExecutor) Executors.newCachedThreadPool(new NodeThreadFactory());
        }
        else
        {
            nodeExecutor = (ThreadPoolExecutor) Executors.newFixedThreadPool(maxNodes, new NodeThreadFactory());
        }

    }


    /* (non-Javadoc)
      * @see org.jetel.graph.runtime.IThreadManager#initWatchDog(org.jetel.graph.runtime.WatchDog)
      */
    public synchronized void initWatchDog(WatchDog watchDog)
    {
        watchDog.setThreadManager(this);
        watchDog.init();
    }

    /* (non-Javadoc)
      * @see org.jetel.graph.runtime.IThreadManager#executeWatchDog(org.jetel.graph.runtime.WatchDog)
      */
    public synchronized CloverFuture executeWatchDog(WatchDog watchDog)
    {
        return new CloverFutureImpl(watchDog, watchdogExecutor.submit(watchDog));
    }

    public synchronized void executeWatchDog(WatchDog watchDog, FutureCallback<Result> callback)
    {

        watchdogExecutor.submit(new NotifyingFutureTask<Result>(watchDog, callback));
    }

    /* (non-Javadoc)
      * @see org.jetel.graph.runtime.IThreadManager#executeNode(java.lang.Runnable)
      */
    public synchronized void executeNode(Runnable node)
    {
        nodeExecutor.execute(node);
        runningNodes++;
    }

    /* (non-Javadoc)
      * @see org.jetel.graph.runtime.IThreadManager#execute(java.lang.Runnable)
      */
    public void execute(Runnable runnable)
    {
        throw new UnsupportedOperationException("Execute is not supported");
    }

    /* (non-Javadoc)
 * @see org.jetel.graph.runtime.IThreadManager#execute(java.lang.Runnable)
 */
    @Override
    public <T> Future<T> execute(Runnable runnable, T result)
    {
        FutureTask<T> futureTask = new FutureTask<T>(runnable, result);
        Thread thread = new Thread(futureTask, runnable.getClass().getName());
        thread.setContextClassLoader(runnable.getClass().getClassLoader());
        thread.setPriority(Thread.MIN_PRIORITY);
        thread.setDaemon(false);
        thread.start();

        return futureTask;
    }

    /* (non-Javadoc)
     * @see org.jetel.graph.runtime.IThreadManager#execute(java.lang.Runnable)
     */
    @Override
    public <T> Future<T> execute(Callable<T> callable)
    {
        FutureTask<T> futureTask = new FutureTask<T>(callable);
        Thread thread = new Thread(futureTask, callable.getClass().getName());
        thread.setContextClassLoader(callable.getClass().getClassLoader());
        thread.setPriority(Thread.MIN_PRIORITY);
        thread.setDaemon(false);
        thread.start();

        return futureTask;
    }

    /* (non-Javadoc)
      * @see org.jetel.graph.runtime.IThreadManager#getFreeThreadsCount()
      */
    public synchronized int getFreeThreadsCount()
    {
        if (maxNodes > 0)
        {
            return maxNodes - runningNodes;
        }
        else
        {
            return Integer.MAX_VALUE;
        }
    }

    /* (non-Javadoc)
      * @see org.jetel.graph.runtime.IThreadManager#releaseNodeThreads(int)
      */
    public synchronized void releaseNodeThreads(int nodeThreadsToRelease)
    {
        runningNodes -= nodeThreadsToRelease;
    }

    /**
     * Waits for all currently running graphs are already done
     * and finishes graph executor life cycle.
     * New graphs cannot be submitted after free invocation.
     */
    public void free()
    {
        watchdogExecutor.shutdown();
        nodeExecutor.shutdown();

        logger.info("DataMapperThreadManager releasing resources");
    }

    /**
     * Immediately finishes graph executor life cycle. All running
     * graphs are aborted.
     */
    public void freeNow()
    {
        watchdogExecutor.shutdownNow();
        nodeExecutor.shutdownNow();

        logger.info("DataMapperThreadManager releasing resources now");
    }


    /**
     * {@link FutureTask} decorated by {@link WatchDog} instance.
     */
    private class CloverFutureImpl implements CloverFuture
    {

        private WatchDog watchDog;
        private Future<Result> future;

        public CloverFutureImpl(WatchDog watchDog, Future<Result> future)
        {

            this.watchDog = watchDog;
            this.future = future;
        }

        @Override
        public WatchDog getWatchDog()
        {
            return watchDog;
        }

        @Override
        public boolean cancel(boolean b)
        {
            return future.cancel(b);
        }

        @Override
        public boolean isCancelled()
        {
            return future.isCancelled();
        }

        @Override
        public boolean isDone()
        {
            return future.isDone();
        }

        @Override
        public Result get() throws InterruptedException, ExecutionException
        {
            return future.get();
        }

        @Override
        public Result get(long l, TimeUnit timeUnit) throws InterruptedException, ExecutionException, TimeoutException
        {
            return future.get(l, timeUnit);
        }
    }


    /**
     * Thread factory for watchdog thread pool.
     */
    private static class WatchdogThreadFactory implements ThreadFactory
    {

        public Thread newThread(Runnable r)
        {
            return new Thread(r, "DataMapperGraphWatchDog");
        }

    }

    /**
     * Thread factory for node thread pool.
     */
    private static class NodeThreadFactory implements ThreadFactory
    {

        public Thread newThread(Runnable r)
        {
            Thread thread = new Thread(r, "Node");
            thread.setDaemon(true);
            thread.setPriority(1);
            return thread;
        }

    }
}
