/**
 *
 * (c) 2011 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */

package com.mulesoft.mule.module.datamapper.clover.impl;

import com.mulesoft.mule.module.datamapper.clover.api.GraphRuntimeContextFactory;

import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.net.URL;
import java.util.Arrays;
import java.util.List;
import java.util.Properties;

import org.apache.commons.beanutils.BeanUtils;
import org.apache.log4j.Level;
import org.jetel.graph.runtime.ExecutionType;
import org.jetel.graph.runtime.GraphRuntimeContext;
import org.jetel.graph.runtime.IAuthorityProxy;
import org.jetel.graph.runtime.PrimitiveAuthorityProxy;

public class GraphRuntimeContextFactoryImpl implements
        GraphRuntimeContextFactory
{

    private static final String configResource = "clover-graph-runtime-config.properties";
    public static final String DATAMAPPER_DEBUG_MODE = "datamapper.debugMode";
    public static final String DATAMAPPER_DEBUG_DIRECTORY = "datamapper.debugDirectory";
    public static final String DATAMAPPER_USE_JMX = "datamapper.useJMX";


    private Properties properties;
    private static IAuthorityProxy authorityProxy = new PrimitiveAuthorityProxy();

    private final List<String> propertyNames = Arrays.asList(
            "trackingInterval",
            "skipCheckConfig",
            "verboseMode",
            "useJMX",
            "waitForJMXClient",
            "password",
            "debugMode",
            "debugDirectory",
            "runtimeClassPath",
            "compileClassPath",
            "synchronizedRun",
            "transactionMode",
            "batchMode",
            "contextURL"
    );
    private ExecutionType executionType;
    private ClassLoader classLoader;

    public GraphRuntimeContextFactoryImpl(ExecutionType executionType){
        this(executionType,Thread.currentThread().getContextClassLoader());
    }

    public GraphRuntimeContextFactoryImpl(ExecutionType executionType, ClassLoader classLoader)
    {
        this.executionType = executionType;
        this.classLoader = classLoader;
        this.properties = new Properties();
        URL propertiesUrl = classLoader.getResource(configResource);
        if (propertiesUrl != null)
        {
            try
            {
                properties.load(propertiesUrl.openStream());
            }
            catch (IOException e)
            {

            }
        }

        loadFromSysProps();


    }

    private void loadFromSysProps()
    {
        //Check system properties for user configuration
        if (Boolean.getBoolean(DATAMAPPER_DEBUG_MODE))
        {
            properties.setProperty("debugMode", "true");
        }
        else
        {
            properties.setProperty("debugMode", "false");
        }

        if (System.getProperty(DATAMAPPER_DEBUG_DIRECTORY) != null)
        {
            properties.setProperty("debugDirectory", System.getProperty(DATAMAPPER_DEBUG_DIRECTORY));
        }

        if (Boolean.getBoolean(DATAMAPPER_USE_JMX))
        {
            properties.setProperty("useJMX", "true");
        }
        else
        {
            //use default properties
            properties.setProperty("useJMX", "false");
        }
    }


    @Override
    public GraphRuntimeContext createGraphRuntimeContext()
    {
        return createGraphRuntimeContext(new Properties());
    }

    @Override
    public GraphRuntimeContext createGraphRuntimeContext(Properties properties)
    {

        GraphRuntimeContext context = new GraphRuntimeContext();
        context.setExecutionType(executionType);
        context.setSkipCheckConfig(true);
        context.setAuthorityProxy(authorityProxy);
        context.setDebugMode(false);
        context.setUseJMX(false);

        for (String property : propertyNames)
        {
            if (this.properties.containsKey(property))
            {
                try
                {
                    BeanUtils.setProperty(context, property, this.properties.get(property));
                }
                catch (IllegalAccessException e)
                {
                    throw new RuntimeException(e);
                }
                catch (InvocationTargetException e)
                {
                    throw new RuntimeException(e);
                }
            }
        }
        if (Boolean.getBoolean(DATAMAPPER_DEBUG_MODE))
        {
            context.setVerboseMode(true);
            context.setLogLevel(Level.INFO);
        }
        else
        {
            context.setVerboseMode(false);
            context.setLogLevel(Level.WARN);
            context.setTrackingInterval(Integer.MAX_VALUE);
        }


        if (properties != null)
        {
            context.addAdditionalProperties(properties);
        }
        context.setClassLoader(classLoader);

        return context;
    }
}
