/**
 *
 * (c) 2011 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */

package com.mulesoft.mule.module.datamapper.impl;

import com.mulesoft.mule.module.datamapper.api.FutureCallback;
import com.mulesoft.mule.module.datamapper.api.GraphExecutor;
import com.mulesoft.mule.module.datamapper.api.GraphProvider;
import com.mulesoft.mule.module.datamapper.api.OutputArgumentHandler;
import com.mulesoft.mule.module.datamapper.api.Status;
import com.mulesoft.mule.module.datamapper.api.TransformationEngine;
import com.mulesoft.mule.module.datamapper.api.TransformationResult;
import com.mulesoft.mule.module.datamapper.api.exception.DataMapperExecutionException;
import com.mulesoft.mule.module.datamapper.clover.MappingFormat;
import com.mulesoft.mule.module.datamapper.util.BlockingPipeList;

import java.io.Closeable;
import java.io.IOException;
import java.io.PipedInputStream;
import java.io.PipedOutputStream;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import org.apache.commons.lang.Validate;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * This is a default implementation of a graph Executor.
 */
public class DefaultGraphExecutor<T> implements GraphExecutor
{

    public static final String MULE_INPUT_PAYLOAD_KEY = "inputPayload";
    public static final String MULE_OUTPUT_PAYLOAD_KEY = "outputPayload";


    private static Log logger = LogFactory.getLog(DefaultGraphExecutor.class);


    private GraphProvider<T> graphProvider;
    private TransformationEngine<T> engine;
    private GraphProvider<T> asyncGraphProvider;

    public static <T> DefaultGraphExecutor<T> createGraphExecutor(GraphProvider<T> graphProvider,
                                                                  GraphProvider<T> asyncGraphProvider,
                                                                  TransformationEngine<T> engine)
    {
        return new DefaultGraphExecutor<T>(graphProvider, asyncGraphProvider, engine);
    }

    private DefaultGraphExecutor(GraphProvider<T> graphProvider, GraphProvider<T> asyncGraphProvider, TransformationEngine<T> engine)
    {
        Validate.notNull(graphProvider);
        this.asyncGraphProvider = asyncGraphProvider;
        this.graphProvider = graphProvider;
        this.engine = engine;
    }

    @Override
    public TransformationResult execute(Object input,
                                        Map<String, Object> inputArguments) throws DataMapperExecutionException
    {
        T currentGraph = graphProvider.takeGraph();


        final Map<String, Object> arguments = new HashMap<String, Object>();
        arguments.put(MULE_INPUT_PAYLOAD_KEY, input);
        arguments.put(MULE_OUTPUT_PAYLOAD_KEY, null);     //Set null to output payload

        if (inputArguments != null)
        {
            arguments.putAll(inputArguments);
        }

        TransformationResult result = null;
        try
        {
            result = getEngine().execute(currentGraph, arguments);

            if (logger.isDebugEnabled())
            {
                logger.debug("Graph executed successfully");
            }

            if (result.getStatus() != Status.OK)
            {
                throw new DataMapperExecutionException("Error executing graph: " + result.getMessage(), result.getCauseException());
            }

            return result;
        }
        finally
        {
            if (result == null || result.getStatus() != Status.OK)
            {
                graphProvider.invalidateObject(currentGraph);
            }
            else
            {
                graphProvider.releaseGraph(currentGraph);
            }
        }

    }

    private Iterator<?> executeObjectStreaming(int pipeSize, T currentGraph, Map<String, Object> arguments, OutputArgumentHandler outputArgumentHandler) throws DataMapperExecutionException
    {
        BlockingPipeList<?> outputSource = new BlockingPipeList<Object>(pipeSize);
        arguments.put(MULE_OUTPUT_PAYLOAD_KEY, outputSource);
        try
        {
            getEngine().executeLater(currentGraph, arguments, new GraphExecutionCallback(currentGraph, outputSource, outputArgumentHandler));
            return outputSource.iterator();
        }
        catch (DataMapperExecutionException e)
        {
            asyncGraphProvider.invalidateObject(currentGraph);

            outputSource.close();
            logger.error("Exception while executing transformation graph", e);
            throw e;
        }
        catch (Throwable e)
        {
            asyncGraphProvider.invalidateObject(currentGraph);

            logger.error("Exception while initializing the graph", e);
            throw new RuntimeException(e);
        }
    }

    @Override
    public Object executeStreaming(Object inputStream,
                                   int pipeSize,
                                   Map<String, Object> inputArguments, OutputArgumentHandler outputArgumentHandler) throws DataMapperExecutionException
    {
        T currentGraph;
        final Map<String, Object> arguments = new HashMap<String, Object>();
        arguments.put(MULE_INPUT_PAYLOAD_KEY, inputStream);

        if (inputArguments != null)
        {
            arguments.putAll(inputArguments);
        }
        currentGraph = asyncGraphProvider.takeGraph();
        MappingFormat outputType = getEngine().getOutputType(currentGraph);
        if (outputType.isFileBased())
        {
            return executeFileStreaming(pipeSize, currentGraph, arguments, outputArgumentHandler);
        }
        else
        {
            return executeObjectStreaming(pipeSize, currentGraph, arguments, outputArgumentHandler);
        }
    }



    @Override
    public String getOutputContentType()
    {
        T currentGraph = graphProvider.takeGraph();
        try
        {
            return getEngine().getOutputType(currentGraph).getContentType();
        }
        finally
        {
            graphProvider.releaseGraph(currentGraph);
        }
    }

    @Override
    public String getOutputEncoding()
    {
        T currentGraph = graphProvider.takeGraph();
        try
        {
            return getEngine().getOutputEncoding(currentGraph);
        }
        finally
        {
            graphProvider.releaseGraph(currentGraph);
        }
    }

    private Object executeFileStreaming(int pipeSize, T currentGraph, Map<String, Object> arguments, OutputArgumentHandler outputArgumentHandler) throws DataMapperExecutionException
    {
        final PipedOutputStream outputSink = new PipedOutputStream();
        arguments.put(MULE_OUTPUT_PAYLOAD_KEY, outputSink);

        PipedInputStream inputSource = null;
        try
        {
            inputSource = new PipedInputStream(outputSink, pipeSize);
            //Do not specify input source clover will close it
            getEngine().executeLater(currentGraph, arguments, new GraphExecutionCallback(currentGraph, null, outputArgumentHandler));
            return inputSource;
        }
        catch (DataMapperExecutionException e)
        {

            asyncGraphProvider.invalidateObject(currentGraph);

            try
            {
                outputSink.close();
            }
            catch (IOException e1)
            {
                logger.error("Error closing outputSink");
            }
            try
            {
                inputSource.close();
            }
            catch (IOException e1)
            {
                logger.error("Error closing inputSource");
            }
            logger.error("Exception while executing transformation graph", e);
            throw e;
        }
        catch (Throwable e)
        {

            asyncGraphProvider.invalidateObject(currentGraph);

            logger.error("Exception while initializing the graph", e);
            throw new RuntimeException(e);
        }
    }


    private class GraphExecutionCallback implements FutureCallback<TransformationResult>
    {

        private final T graph;
        private Closeable closeable;
        private OutputArgumentHandler outputArgumentHandler;
        private Throwable throwable;

        public GraphExecutionCallback(T graph, Closeable closeable, OutputArgumentHandler outputArgumentHandler)
        {
            super();
            this.graph = graph;
            this.closeable = closeable;
            this.outputArgumentHandler = outputArgumentHandler;
        }

        @Override
        public void done(TransformationResult result)
        {
            Map<String, Object> otherResults = result.getOutputArguments();
            for (Map.Entry<String, Object> outputArgument : otherResults.entrySet())
            {
                outputArgumentHandler.addArgument(outputArgument.getKey(), outputArgument.getValue());
            }
            release();

        }

        @Override
        public void failed(Throwable throwable)
        {
            this.throwable = throwable;
            logger.error("Error while executing graph ", throwable);
            release();
        }

        @Override
        public void cancelled()
        {
            release();
        }

        private void release()
        {
            if (throwable != null)
            {
                asyncGraphProvider.invalidateObject(graph);
            }
            else
            {
                asyncGraphProvider.releaseGraph(graph);
            }
            try
            {
                if (closeable != null)
                {
                    closeable.close();
                }
            }
            catch (IOException e)
            {
                logger.error("Error while closing  ", e);
            }
        }
    }

    public TransformationEngine<T> getEngine()
    {
        return engine;
    }
}
