package com.mulesoft.mule.module.datamapper.processors;

import org.mule.api.DefaultMuleException;
import org.mule.api.MessagingException;
import org.mule.api.MuleContext;
import org.mule.api.MuleEvent;
import org.mule.api.MuleException;
import org.mule.api.context.MuleContextAware;
import org.mule.api.expression.ExpressionManager;
import org.mule.api.lifecycle.Disposable;
import org.mule.api.lifecycle.Initialisable;
import org.mule.api.lifecycle.InitialisationException;
import org.mule.api.processor.MessageProcessor;
import org.mule.api.transformer.DataType;
import org.mule.api.transformer.Transformer;
import org.mule.api.transformer.TransformerException;
import org.mule.config.i18n.CoreMessages;
import org.mule.module.xml.transformer.XmlToOutputHandler;
import org.mule.transformer.simple.ObjectToString;
import org.mule.transformer.types.DataTypeFactory;
import org.mule.transport.NullPayload;
import org.mule.util.AttributeEvaluator;

import com.mulesoft.mule.module.datamapper.DataMapperModule;
import com.mulesoft.mule.module.datamapper.api.exception.DataMapperCreationException;
import com.mulesoft.mule.module.datamapper.api.exception.DataMapperExecutionException;
import com.mulesoft.mule.module.datamapper.boot.DataMapperLicenseCheck;
import com.mulesoft.mule.module.datamapper.i18n.DataMapperMessages;
import com.mulesource.licm.LicenseKeyException;

import java.util.HashMap;
import java.util.Map;

import javax.xml.stream.XMLStreamReader;

import de.schlichtherle.license.LicenseContentException;
import org.apache.commons.lang.StringUtils;


/**
 * DataMapperMessageProcessor invokes the {@link com.mulesoft.mule.module.datamapper.DataMapperModule#transform(org.mule.api.MuleEvent, Object, java.util.Map, boolean, int)}  method in {@link com.mulesoft.mule.module.datamapper.DataMapperModule }.
 * For each argument there is a field in this processor to match it.
 * Before invoking the actual method the processor will evaluate and transform where possible to the expected argument type.
 */
public class DataMapperMessageProcessor
        implements MuleContextAware, Initialisable, MessageProcessor, Disposable
{


    public static final String CONTENT_TYPE = "Content-Type";
    private DataMapperModule module;
    private Map<String, String> inputArguments;

    private DataMapperGraphConfig config;
    private String input;
    private boolean stream = false;
    private int pipeSize = 4024;
    private String returnClass;

    private Class<?> returnClassType;

    private String target;
    private MuleContext muleContext;

    public DataMapperMessageProcessor()
    {
        this.module = new DataMapperModule();
        this.inputArguments = new HashMap<String, String>();
    }


    @Override
    public void initialise() throws InitialisationException
    {

        try
        {
            DataMapperLicenseCheck.checkDMEntitlement();
        }
        catch (LicenseContentException e)
        {
            throw new InitialisationException(new DataMapperMessages().createNoLicenseFound(), e, this);
        }
        catch (LicenseKeyException e)
        {
            throw new InitialisationException(new DataMapperMessages().createNoLicenseFound(), e, this);
        }
        if (getConfig() == null)
        {
            final Map<String, DataMapperGraphConfig> configMap = this.muleContext.getRegistry().lookupByType(DataMapperGraphConfig.class);
            if (configMap.size() != 1)
            {
                throw new InitialisationException(new DataMapperMessages().createNoDefaultConfig(configMap.size()), this);
            }
            setConfig(configMap.values().iterator().next());
        }
        if (returnClass != null)
        {
            try
            {
                returnClassType = Class.forName(returnClass);
            }
            catch (ClassNotFoundException e)
            {
                throw new InitialisationException(new DataMapperMessages().createReturnClassError(), e, this);
            }
        }
        module.setTransformationGraphPath(getConfig().getTransformationGraphPath());
        module.setInitialPoolSize(getConfig().getInitialPoolSize());
        module.initialise();
    }

    @Override
    public MuleEvent process(MuleEvent event) throws MuleException
    {

        final ExpressionManager expressionManager = muleContext.getExpressionManager();
        final Map<String, Object> resolvedInputArguments = new HashMap<String, Object>();
        for (Map.Entry<String, String> inputArgumentDef : inputArguments.entrySet())
        {
            AttributeEvaluator attributeEvaluator = new AttributeEvaluator(inputArgumentDef.getValue());
            attributeEvaluator.initialize(expressionManager);
            resolvedInputArguments.put(inputArgumentDef.getKey(), attributeEvaluator.resolveValue(event.getMessage()));
        }
        Object dmResult;
        try
        {
            final AttributeEvaluator inputEvaluator = new AttributeEvaluator(getInput());
            inputEvaluator.initialize(expressionManager);
            Object inputObject = inputEvaluator.resolveValue(event.getMessage());
            inputObject = transformInputIfRequired(inputObject);
            dmResult = module.transform(event, inputObject, resolvedInputArguments, stream, pipeSize);
            dmResult = transformOutputIfRequired(event, dmResult);
            if (hasTargetDefined())
            {
                muleContext.getExpressionManager().enrich(target, event, dmResult);
            }
            else
            {
                event.getMessage().setPayload(dmResult);
            }

            if (!hasTargetDefined())
            {
                final String outputContentType = module.getOutputContentType();
                if (!StringUtils.isEmpty(outputContentType))
                {
                    event.getMessage().setOutboundProperty(CONTENT_TYPE, outputContentType);
                }


                final String outputEncoding = module.getOutputEncoding();
                if (!StringUtils.isEmpty(outputEncoding))
                {
                    event.getMessage().setEncoding(outputEncoding);
                }
            }

            return event;
        }
        catch (DataMapperExecutionException e)
        {
            throw new DataMapperMessageExecutionException(event, e);
        }
        catch (DataMapperCreationException e)
        {
            throw new MessagingException(event, e);
        }
    }

    private Object transformOutputIfRequired(MuleEvent event, Object dmResult) throws TransformerException
    {
        if (returnClassType != null)
        {
            DataType source = DataTypeFactory.createFromObject(dmResult);
            DataType<?> resultType = DataTypeFactory.create(returnClassType);

            // If no conversion is necessary, just return the payload as-is
            if (!resultType.isCompatibleWith(source))
            {


                // The transformer to execute on this message
                Transformer transformer = muleContext.getRegistry().lookupTransformer(source, resultType);
                if (transformer == null)
                {
                    throw new TransformerException(CoreMessages.noTransformerFoundForMessage(source, resultType));
                }

                // Pass in the message itself
                dmResult = transformer.transform(dmResult, event.getEncoding());
            }

        }
        return dmResult;
    }

    private Object transformInputIfRequired(Object inputObject) throws org.mule.api.registry.RegistrationException, DefaultMuleException, TransformerException
    {
        if (inputObject instanceof XMLStreamReader)
        {
            //Todo replace all this with the mule way but for some reason is not working. Temporary Hack
            // The transformer to execute on this message
            XmlToOutputHandler xmlToOutputHandler = muleContext.getRegistry().lookupObject(XmlToOutputHandler.class);
            if (xmlToOutputHandler == null)
            {
                throw new DefaultMuleException("Missing transformer XmlToOutputHandler");
            }
            Object transform = xmlToOutputHandler.transform(inputObject);
            inputObject = new ObjectToString().transform(transform);
        } else if(inputObject == null) {
            return NullPayload.getInstance();
        }
        return inputObject;
    }

    private boolean hasTargetDefined()
    {
        return !StringUtils.isEmpty(target) && !isPayloadExpression(target);
    }

    private boolean isPayloadExpression(String target)
    {
        return "#[payload]".equals(target);
    }

    @Override
    public void setMuleContext(MuleContext context)
    {
        this.muleContext = context;
        this.module.setMuleContext(context);
    }


    public String getTarget()
    {
        return target;
    }

    public void setTarget(String target)
    {
        this.target = target;
    }

    public Map<String, String> getInputArguments()
    {
        return inputArguments;
    }

    public void setInputArguments(Map<String, String> inputArguments)
    {
        this.inputArguments = inputArguments;
    }

    public DataMapperGraphConfig getConfig()
    {
        return config;
    }

    public void setConfig(DataMapperGraphConfig config)
    {
        this.config = config;
    }

    public String getInput()
    {
        return input;
    }

    public void setInput(String input)
    {
        this.input = input;
    }

    public boolean isStream()
    {
        return stream;
    }

    public void setStream(boolean stream)
    {
        this.stream = stream;
    }

    public int getPipeSize()
    {
        return pipeSize;
    }

    public void setPipeSize(int pipeSize)
    {
        this.pipeSize = pipeSize;
    }

    public String getReturnClass()
    {
        return returnClass;
    }

    public void setReturnClass(String returnClass)
    {
        this.returnClass = returnClass;
    }

    @Override
    public void dispose()
    {
        if (module != null)
        {
            module.dispose();
        }
    }
}
