/**
 *
 * (c) 2013 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.module.datamapper.processors;

import org.mule.api.DefaultMuleException;
import org.mule.api.MuleEvent;
import org.mule.api.MuleException;
import org.mule.api.lifecycle.InitialisationException;
import org.mule.api.processor.MessageProcessor;
import org.mule.api.processor.MessageProcessors;
import org.mule.config.i18n.MessageFactory;
import org.mule.processor.AbstractMessageProcessorOwner;
import org.mule.processor.chain.DefaultMessageProcessorChainBuilder;

import java.io.Closeable;
import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;
import java.util.List;

public class TryResource extends AbstractMessageProcessorOwner implements MessageProcessor
{

    private List<MessageProcessor> messageProcessors;

    private MessageProcessor headMP;
    private MessageProcessor tailMP;

    private boolean suppressedException;


    @Override
    public void initialise() throws InitialisationException
    {

        if (messageProcessors.isEmpty())
        {
            throw new InitialisationException(MessageFactory.createStaticMessage("Try resource must have at least one inner message processor."), this);
        }
        headMP = MessageProcessors.singletonChain(messageProcessors.get(0));
        DefaultMessageProcessorChainBuilder defaultMessageProcessorChainBuilder = new DefaultMessageProcessorChainBuilder();
        defaultMessageProcessorChainBuilder.chain(messageProcessors.subList(1, messageProcessors.size()));
        try
        {
            tailMP = defaultMessageProcessorChainBuilder.build();
        }
        catch (MuleException e)
        {
            throw new InitialisationException(e, this);
        }
        super.initialise();

    }

    @Override
    protected List<MessageProcessor> getOwnedMessageProcessors()
    {
        return messageProcessors;
    }

    @Override
    public MuleEvent process(MuleEvent event) throws MuleException
    {
        Closeable resource = null;
        try
        {
            MuleEvent process = headMP.process(event);
            Object payload = process.getMessage().getPayload();
            resource = asCloseable(payload);
            return tailMP.process(process);
        }
        finally
        {
            try
            {
                if (resource != null)
                {
                    resource.close();
                }
            }
            catch (IOExceptionAdapter e)
            {
                if (!isSuppressedException())
                {
                    throw new DefaultMuleException(e.getCause());
                }
            }
            catch (IOException e)
            {
                if (!isSuppressedException())
                {
                    throw new DefaultMuleException(e);
                }

            }
        }

    }

    private Closeable asCloseable(Object payload) throws DefaultMuleException
    {
        Closeable resource;
        if (payload instanceof Closeable)
        {
            resource = (Closeable) payload;
        }
        else
        {
            try
            {
                Method close = payload.getClass().getMethod("close");
                resource = new CloseableProxyClass(payload, close);
            }
            catch (NoSuchMethodException e)
            {
                throw new DefaultMuleException("Payload must be instance of " + Closeable.class + " or at least have a close() method declare");
            }
        }

        return resource;
    }

    public boolean isSuppressedException()
    {
        return suppressedException;
    }

    public void setSuppressedException(boolean suppressedException)
    {
        this.suppressedException = suppressedException;
    }

    public void setMessageProcessors(List<MessageProcessor> messageProcessors)
    {
        this.messageProcessors = messageProcessors;
    }

    public void setHeadMP(MessageProcessor headMP)
    {
        this.headMP = headMP;
    }

    public void setTailMP(MessageProcessor tailMP)
    {
        this.tailMP = tailMP;
    }

    private class CloseableProxyClass implements Closeable
    {

        private Object toDelegate;
        private Method closeMethod;

        private CloseableProxyClass(Object toDelegate, Method closeMethod)
        {
            this.toDelegate = toDelegate;
            this.closeMethod = closeMethod;
        }


        @Override
        public void close() throws IOException
        {
            try
            {
                closeMethod.invoke(toDelegate);
            }
            catch (IllegalAccessException e)
            {
                throw new IOExceptionAdapter(e);
            }
            catch (InvocationTargetException e)
            {
                throw new IOExceptionAdapter(e.getCause());
            }
        }
    }

    private class IOExceptionAdapter extends IOException
    {

        private IOExceptionAdapter(Throwable throwable)
        {
            super(throwable);
        }
    }
}
