/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.mtf.tools.api.queue;

import static java.util.Objects.isNull;

import java.io.Serializable;
import java.util.HashMap;
import java.util.Map;

import javax.inject.Inject;
import javax.inject.Named;

import org.mule.munit.mtf.tools.internal.error.QueueTimeOutException;
import org.mule.munit.runner.component.rules.TestRule;
import org.mule.runtime.api.lifecycle.Initialisable;
import org.mule.runtime.core.api.config.MuleProperties;
import org.mule.runtime.core.api.util.queue.DefaultQueueConfiguration;
import org.mule.runtime.core.api.util.queue.Queue;
import org.mule.runtime.core.api.util.queue.QueueManager;

/**
 * Rule to manage temporary queues
 *
 * @author Mulesoft Inc.
 * @since 1.0.0
 */
public class TemporaryQueueRule implements TestRule {

  @Inject
  @Named(MuleProperties.OBJECT_QUEUE_MANAGER)
  protected QueueManager queueManager;

  private Map<String, Queue> queues = new HashMap<>();

  public void push(Serializable value, String queueName) {
    executeOnQueue(queueName, queue -> {
      queue.put(value);
      return null;
    });
  }

  public Serializable pop(String queueName, Long timeOut) {
    if (isNull(timeOut)) {
      return executeOnQueue(queueName, Queue::take);
    } else {
      return executeOnQueue(queueName, queue -> queue.poll(timeOut));
    }
  }

  public int size(String queueName) {
    return executeOnQueue(queueName, Queue::size);
  }

  @Override
  public void reset() {
    queues.keySet().forEach(q -> executeOnQueue(q, queue -> {
      queue.clear();
      return null;
    }));
  }

  private <T> T executeOnQueue(String queue, TemporaryQueueTask<T> task) {
    try {
      return task.run(queues.computeIfAbsent(queue, (queueName) -> {
        queueManager.setQueueConfiguration(queueName, new DefaultQueueConfiguration());
        return queueManager.getQueueSession().getQueue(queueName);
      }));
    } catch (InterruptedException e) {
      throw new QueueTimeOutException(e);
    }
  }

  @FunctionalInterface
  private interface TemporaryQueueTask<T> {

    T run(Queue queue) throws InterruptedException;
  }

}
