/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.mtf.tools.internal.queue;

import static java.util.Optional.ofNullable;
import static org.mule.runtime.api.meta.ExpressionSupport.NOT_SUPPORTED;

import java.io.Serializable;
import java.util.concurrent.TimeUnit;

import javax.inject.Inject;

import org.mule.munit.mtf.tools.api.queue.TemporaryQueueRule;
import org.mule.munit.mtf.tools.internal.error.QueueTimeOutException;
import org.mule.runtime.api.metadata.DataType;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.extension.api.annotation.Expression;
import org.mule.runtime.extension.api.annotation.Ignore;
import org.mule.runtime.extension.api.annotation.error.Throws;
import org.mule.runtime.extension.api.annotation.param.Content;
import org.mule.runtime.extension.api.annotation.param.Optional;
import org.mule.runtime.extension.api.runtime.operation.Result;

/**
 * Operations to store values on a temporary queue
 *
 * @author Mulesoft Inc.
 * @since 1.0.0
 */
public class QueueOperations {

  public static final String DEFAULT_QUEUE_NAME = "MTF_QUEUE";

  @Inject
  protected TemporaryQueueRule temporaryQueueRule;

  /**
   * Pushes the given value to a temporary queue so this can be consumed later on demand
   *
   * @param value     Value to queue
   * @param queueName The name of the queue where dequeue a value
   */
  @Throws(QueueErrorTypeProvider.class)
  public void queue(@Content TypedValue<Serializable> value,
                    @Optional String queueName) {
    temporaryQueueRule.push(value, getQueueName(queueName));
  }

  /**
   * Pops the given data from a temporary queue.
   *
   * @param queueName   The name of the queue where dequeue a value
   * @param timeout     Time to wait for a value until the connector throws a QUEUE_TIMEOUT error
   * @param timeoutUnit Time Unit to use to wait for a value to arrive
   * @return The next queued value
   */
  @Throws(QueueErrorTypeProvider.class)
  public Result<Serializable, Void> dequeue(@Optional String queueName,
                                            @Optional(defaultValue = "10000") Long timeout,
                                            @Optional(defaultValue = "MILLISECONDS") TimeUnit timeoutUnit) {
    return ofNullable(temporaryQueueRule.pop(getQueueName(queueName), timeoutUnit.toMillis(timeout))).map(this::asResult)
        .orElseThrow(QueueTimeOutException::new);
  }

  private Result<Serializable, Void> asResult(Object value) {
    TypedValue<Serializable> typedValue = value instanceof TypedValue
        ? (TypedValue<Serializable>) value
        : new TypedValue<>((Serializable) value, DataType.fromType(value.getClass()));

    return Result.<Serializable, Void>builder()
        .output(typedValue.getValue())
        .mediaType(typedValue.getDataType().getMediaType())
        .build();
  }

  @Ignore
  static String getQueueName(String queueName) {
    return queueName != null
        ? queueName
        : DEFAULT_QUEUE_NAME;
  }
}
