/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.mtf.tools.api;

import static com.google.common.collect.Sets.newHashSet;
import static org.mule.metadata.java.api.JavaTypeLoader.JAVA;
import static org.mule.runtime.api.meta.Category.COMMUNITY;

import org.mule.metadata.api.ClassTypeLoader;
import org.mule.metadata.api.builder.BaseTypeBuilder;
import org.mule.metadata.api.model.MetadataType;
import org.mule.runtime.api.meta.MuleVersion;
import org.mule.runtime.api.meta.model.XmlDslModel;
import org.mule.runtime.api.meta.model.declaration.fluent.ConstructDeclarer;
import org.mule.runtime.api.meta.model.declaration.fluent.ExtensionDeclarer;
import org.mule.runtime.api.meta.model.declaration.fluent.OperationDeclarer;
import org.mule.runtime.api.meta.model.declaration.fluent.OptionalParameterDeclarer;
import org.mule.runtime.api.meta.model.declaration.fluent.ParameterGroupDeclarer;
import org.mule.runtime.api.meta.model.stereotype.StereotypeModel;
import org.mule.runtime.api.meta.model.stereotype.StereotypeModelBuilder;
import org.mule.runtime.extension.api.declaration.type.ExtensionsTypeLoaderFactory;
import org.mule.runtime.extension.api.loader.ExtensionLoadingContext;
import org.mule.runtime.extension.api.loader.ExtensionLoadingDelegate;

/**
 * <p>
 * Provides the MUnit Extension Model for the "crafted" extensionModelLoaderDescriptor
 * </p>
 *
 * @author Mulesoft Inc.
 * @since 1.0.0
 */
public class MtfExtensionLoadingDelegate implements ExtensionLoadingDelegate {

  // General Attributes
  private static final String VERSION = "1.0.0-SNAPSHOT";
  private static final String DESCRIPTION = "MTF: Tools";
  private static final String NAME = "MTF Tools";
  private static final String VENDOR = "MuleSoft, Inc.";
  private static final MuleVersion MIN_MULE_VERSION = new MuleVersion("4.2.0");
  // XML DSL Attributes
  private static final String PREFIX = "mtf";
  private static final String NAMESPACE = "http://www.mulesoft.org/schema/mule/mtf";
  private static final String SCHEMA_LOCATION = "http://www.mulesoft.org/schema/mule/mtf/current/mule-mtf.xsd";
  private static final String SCHEMA_VERSION = VERSION;
  private static final String XSD_FILE_NAME = "mule-mtf.xsd";
  // Operations - Test
  private static final String TOOLING_TEST_NAME = "tooling-test";
  private static final String TOOLING_TEST_DESCRIPTION = "The Tooling test";
  private static final String DESCRIPTION_NAME = "description";
  private static final String DESCRIPTION_DESCRIPTION = "Description message to be shown in case of test failure.";
  private static final String IGNORE_NAME = "ignore";
  private static final String IGNORE_DESCRIPTION = "Defines if the test must be ignored.";
  private static final String TAGS_NAME = "tags";
  private static final String TAGS_DESCRIPTION = "Defines the tags for the test.";
  private static final String EXPECTED_FAILURE_CODE_NAME = "expectFailureCode";
  private static final String EXPECTED_FAILURE_CODE_DESCRIPTION = "Failure code expected on this test.";
  private static final String EXPECTED_FAILURE_MESSAGE_NAME = "expectFailureMessage";
  private static final String EXPECTED_FAILURE_MESSAGE_DESCRIPTION = "Failure message expected on this test.";
  private static final String GET_METADATA_KEYS = "get-metadata-keys";
  private static final String GET_METADATA_KEYS_DESCRIPTION = "Gets the metadata keys of the component";
  private static final String GET_OUTPUT_METADATA = "get-output-metadata";
  private static final String GET_OUTPUT_METADATA_DESCRIPTION = "Gets the output metadata of the component";
  private static final String GET_ATTRIBUTES_METADATA = "get-attributes-metadata";
  private static final String GET_ATTRIBUTES_DESCRIPTION = "Gets the attributes metadata of the component";
  private static final String GET_INPUT_METADATA = "get-input-metadata";
  private static final String GET_INPUT_METADATA_DESCRIPTION = "Gets the input metadata of the component's parameter";
  private static final String VALIDATION_NAME = "validation";
  private static final String VALIDATION_DESCRIPTION = "Processors that define validation of the test";
  private static final String NESTED_COMPONENT_NAME = "component";
  private static final String NESTED_COMPONENT_DESCRIPTION = "Component over which metadata is obtained";
  private static final String METADATA_SCOPE_NAME = "metadataScope";
  private static final String METADATA_SCOPE_DESCRIPTION = "Scope that calculates metadata over a component";
  private static final String INPUT_METADATA_PARAMETER = "parameter";
  private static final String INPUT_METADATA_PARAMETER_DESCRIPTION = "Component's parameter over which to calculate metadata";
  private static final String ASSERT_TYPE_PROCESSOR = "assert-type";
  private static final String ASSERT_TYPE_PROCESSOR_DESCRIPTION = "Asserts if two Metadata types are equal";
  private static final String ACTUAL_PARAMETER = "actual";
  private static final String ACTUAL_PARAMETER_DESCRIPTION = "Actual metadata type to compare";
  private static final String FROM_CLASS_PARAMETER = "fromClass";
  private static final String FROM_CLASS_PARAMETER_DESCRIPTION = "Compares to a Metadata Type obtained from a Java class";
  private static final String FROM_SCHEMA_PARAMETER = "fromSchema";
  private static final String FROM_SCHEMA_PARAMETER_DESCRIPTION = "Compares to a Metadata Type obtained from a schema file";

  // Stereotypes
  private static final StereotypeModel METADATA_SCOPE_STEREOTYPE =
      StereotypeModelBuilder.newStereotype(METADATA_SCOPE_NAME, PREFIX).build();

  private ClassTypeLoader typeLoader = ExtensionsTypeLoaderFactory.getDefault().createTypeLoader();
  private BaseTypeBuilder typeBuilder = BaseTypeBuilder.create(JAVA);

  @Override
  public void accept(ExtensionDeclarer extensionDeclarer, ExtensionLoadingContext extensionLoadingContext) {
    declareGeneral(extensionDeclarer);
    declareXmlDsl(extensionDeclarer);
    declareToolingTest(extensionDeclarer);
    declareMetadataScopes(extensionDeclarer);
    declareProcessors(extensionDeclarer);
  }

  private void declareGeneral(ExtensionDeclarer extensionDeclarer) {
    extensionDeclarer
        .named(NAME)
        .describedAs(DESCRIPTION)
        .onVersion(VERSION)
        .fromVendor(VENDOR)
        .withCategory(COMMUNITY);
  }

  private void declareXmlDsl(ExtensionDeclarer extensionDeclarer) {
    XmlDslModel xmlDslModel = XmlDslModel.builder()
        .setPrefix(PREFIX)
        .setNamespace(NAMESPACE)
        .setSchemaLocation(SCHEMA_LOCATION)
        .setSchemaVersion(SCHEMA_VERSION)
        .setXsdFileName(XSD_FILE_NAME).build();
    extensionDeclarer.withXmlDsl(xmlDslModel);
  }

  private void declareToolingTest(ExtensionDeclarer extensionDeclarer) {
    ConstructDeclarer testDeclarer = extensionDeclarer.withConstruct(TOOLING_TEST_NAME)
        .describedAs(TOOLING_TEST_DESCRIPTION)
        .allowingTopLevelDefinition();

    addOptionalParameter(DESCRIPTION_NAME, DESCRIPTION_DESCRIPTION, String.class, testDeclarer.onDefaultParameterGroup());
    ParameterGroupDeclarer parameterGroupDeclarer = testDeclarer.onDefaultParameterGroup();
    addOptionalParameter(IGNORE_NAME, IGNORE_DESCRIPTION, Boolean.class, parameterGroupDeclarer);
    addOptionalParameter(TAGS_NAME, TAGS_DESCRIPTION, String.class, parameterGroupDeclarer);
    addOptionalParameter(EXPECTED_FAILURE_CODE_NAME, EXPECTED_FAILURE_CODE_DESCRIPTION, String.class, parameterGroupDeclarer);
    addOptionalParameter(EXPECTED_FAILURE_MESSAGE_NAME, EXPECTED_FAILURE_MESSAGE_DESCRIPTION, String.class,
                         parameterGroupDeclarer);

    testDeclarer.withComponent(METADATA_SCOPE_NAME).describedAs(METADATA_SCOPE_DESCRIPTION)
        .withAllowedStereotypes(METADATA_SCOPE_STEREOTYPE);
    declareTestSection(VALIDATION_NAME, VALIDATION_DESCRIPTION, extensionDeclarer, testDeclarer);
  }

  private void declareTestSection(String name, String description, ExtensionDeclarer extensionDeclarer,
                                  ConstructDeclarer testDeclarer) {
    StereotypeModel stereoType = StereotypeModelBuilder.newStereotype(name, PREFIX).build();
    extensionDeclarer.withConstruct(name).describedAs(description).withStereotype(stereoType).withChain();
    testDeclarer.withOptionalComponent(name).withAllowedStereotypes(stereoType);
  }

  private void declareMetadataScopes(ExtensionDeclarer extensionDeclarer) {
    declareMetadataScope(GET_METADATA_KEYS, GET_METADATA_KEYS_DESCRIPTION, extensionDeclarer);
    declareMetadataScope(GET_OUTPUT_METADATA, GET_OUTPUT_METADATA_DESCRIPTION, extensionDeclarer);
    declareMetadataScope(GET_ATTRIBUTES_METADATA, GET_ATTRIBUTES_DESCRIPTION, extensionDeclarer);
    ConstructDeclarer inputMetadataScope =
        declareMetadataScope(GET_INPUT_METADATA, GET_INPUT_METADATA_DESCRIPTION, extensionDeclarer);
    inputMetadataScope.withComponent(INPUT_METADATA_PARAMETER).describedAs(INPUT_METADATA_PARAMETER_DESCRIPTION);
  }

  private ConstructDeclarer declareMetadataScope(String name, String description, ExtensionDeclarer extensionDeclarer) {
    ConstructDeclarer constructDeclarer =
        extensionDeclarer.withConstruct(name).describedAs(description).withStereotype(METADATA_SCOPE_STEREOTYPE);
    constructDeclarer.withComponent(NESTED_COMPONENT_NAME).describedAs(NESTED_COMPONENT_DESCRIPTION);
    return constructDeclarer;
  }

  private void declareProcessors(ExtensionDeclarer extensionDeclarer) {
    OperationDeclarer operationDeclarer =
        extensionDeclarer.withOperation(ASSERT_TYPE_PROCESSOR).describedAs(ASSERT_TYPE_PROCESSOR_DESCRIPTION);
    operationDeclarer.withOutput().ofType(typeBuilder.voidType().build());
    operationDeclarer.withOutputAttributes().ofType(typeBuilder.nullType().build());
    ParameterGroupDeclarer parameterGroupDeclarer = operationDeclarer.onDefaultParameterGroup();
    parameterGroupDeclarer.withOptionalParameter(ACTUAL_PARAMETER).describedAs(ACTUAL_PARAMETER_DESCRIPTION)
        .ofType(typeBuilder.stringType().build());
    parameterGroupDeclarer.withOptionalParameter(FROM_CLASS_PARAMETER).describedAs(FROM_CLASS_PARAMETER_DESCRIPTION)
        .ofType(typeBuilder.stringType().build());
    parameterGroupDeclarer.withOptionalParameter(FROM_SCHEMA_PARAMETER).describedAs(FROM_SCHEMA_PARAMETER_DESCRIPTION)
        .ofType(typeBuilder.stringType().build());
    parameterGroupDeclarer.withExclusiveOptionals(newHashSet(FROM_CLASS_PARAMETER, FROM_SCHEMA_PARAMETER), true);
  }

  private OptionalParameterDeclarer addOptionalParameter(String name, String description, Class<?> clazz,
                                                         ParameterGroupDeclarer declarer) {
    return addOptionalParameter(name, description, typeLoader.load(clazz), declarer);
  }

  private OptionalParameterDeclarer addOptionalParameter(String name, String description, MetadataType type,
                                                         ParameterGroupDeclarer declarer) {
    return declarer
        .withOptionalParameter(name)
        .describedAs(description)
        .ofType(type);
  }

}
