/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.mtf.tools.internal.tooling;

import static org.apache.commons.lang3.StringUtils.isBlank;
import static org.apache.commons.lang3.StringUtils.isEmpty;
import static org.apache.commons.lang3.StringUtils.isNotBlank;
import static org.mule.munit.runner.model.builders.TestRunFilter.NO_TAG_TOKEN;
import static org.mule.runtime.core.api.lifecycle.LifecycleUtils.disposeIfNeeded;
import static org.mule.runtime.core.api.lifecycle.LifecycleUtils.initialiseIfNeeded;
import static org.mule.runtime.core.api.lifecycle.LifecycleUtils.startIfNeeded;
import static org.mule.runtime.core.api.lifecycle.LifecycleUtils.stopIfNeeded;
import static org.slf4j.LoggerFactory.getLogger;

import java.util.Collections;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.inject.Inject;

import org.apache.commons.lang3.StringUtils;
import org.mule.munit.mtf.tools.internal.tooling.metadata.AbstractMetadataScope;
import org.mule.munit.mtf.tools.internal.tooling.metadata.exception.MetadataTestException;
import org.mule.munit.runner.component.TestComponent;
import org.mule.runtime.api.component.AbstractComponent;
import org.mule.runtime.api.event.Event;
import org.mule.runtime.api.exception.MuleException;
import org.mule.runtime.api.lifecycle.InitialisationException;
import org.mule.runtime.api.lifecycle.Lifecycle;
import org.mule.runtime.api.metadata.resolving.FailureCode;
import org.mule.runtime.core.api.event.CoreEvent;
import org.mule.runtime.core.privileged.PrivilegedMuleContext;
import org.mule.runtime.core.privileged.processor.chain.MessageProcessorChain;
import org.slf4j.Logger;

/**
 * Test to validate tooling behavior
 *
 * @author Mulesoft Inc.
 * @since 1.0.0
 */
public class ToolingTest extends AbstractComponent implements TestComponent, Lifecycle {

  private static Logger LOGGER = getLogger(ToolingTest.class);

  private static final String TAG_SEPARATOR = ",";

  @Inject
  protected PrivilegedMuleContext muleContext;

  private boolean ignore;
  private String name;
  private String description;
  private String tags;
  private String expectFailureCode;
  private String expectFailureMessage;
  private AbstractMetadataScope metadataScope;
  private MessageProcessorChain validation;

  @Override
  public String getName() {
    return name;
  }

  public void setName(String name) {
    this.name = name;
  }

  @Override
  public String getDescription() {
    return description;
  }

  public String getExpectFailureCode() {
    return expectFailureCode;
  }

  public String getExpectFailureMessage() {
    return expectFailureMessage;
  }

  public void setExpectFailureMessage(String expectFailureMessage) {
    this.expectFailureMessage = expectFailureMessage;
  }

  public void setExpectFailureCode(String expectFailureCode) {
    this.expectFailureCode = expectFailureCode;
  }

  @Override
  public Event run(Event event) throws Throwable {
    CoreEvent incomingEvent = (CoreEvent) event;
    CoreEvent eventWithMetadata = processMetadata(incomingEvent);
    if (validation != null) {
      return validation.process(eventWithMetadata);
    } else {
      return eventWithMetadata;
    }
  }

  private CoreEvent processMetadata(CoreEvent incomingEvent) throws MetadataTestException {
    CoreEvent eventWithMetadata = incomingEvent;
    try {
      if (metadataScope != null) {
        eventWithMetadata = metadataScope.process(incomingEvent);
      }
      expectedButDidNotFail();
    } catch (MetadataTestException e) {
      if (isExpectingFailure()) {
        expectFailure(e);
      } else {
        throw e;
      }
    }
    return eventWithMetadata;
  }

  private void expectedButDidNotFail() {
    if (isExpectingFailure()) {
      String expectedMessage = StringUtils.EMPTY;
      if (isNotBlank(expectFailureCode)) {
        expectedMessage += " with the failure code: " + expectFailureCode;
      }
      if (isNotBlank(expectFailureMessage)) {
        if (!isEmpty(expectedMessage)) {
          expectedMessage += " and";
        }
        expectedMessage += " with the message: " + expectFailureMessage;
      }
      throw new AssertionError("Test was expecting to fail" + expectedMessage + " but no error was thrown");
    }
  }

  private boolean isExpectingFailure() {
    return isNotBlank(expectFailureCode) || isNotBlank(expectFailureMessage);
  }

  private void expectFailure(MetadataTestException e) {
    String errorMessage = StringUtils.EMPTY;
    if (isNotBlank(expectFailureCode)) {
      FailureCode actualFailureCode = e.getMetadataFailure().getFailureCode();
      if (!actualFailureCode.equals(new FailureCode(expectFailureCode))) {
        errorMessage += " failure code [" + expectFailureCode + "] but was [" + actualFailureCode.getName() + "]";
      }
    }
    if (isNotBlank(expectFailureMessage)) {
      String actualFailureMessage = e.getMetadataFailure().getMessage();
      if (!actualFailureMessage.contains(expectFailureMessage)) {
        if (!isEmpty(errorMessage)) {
          errorMessage += " and";
        }
        errorMessage += " error message [" + expectFailureMessage + "] but was [" + actualFailureMessage + "]";
      }
    }
    if (!isEmpty(errorMessage)) {
      throw new AssertionError("Expected" + errorMessage);
    }
  }

  public void setDescription(String description) {
    this.description = description;
  }

  @Override
  public Set<String> getTags() {
    if (isBlank(tags)) {
      return Collections.emptySet();
    }
    Set<String> tagSet = Stream.of(tags.split(TAG_SEPARATOR)).collect(Collectors.toSet());
    if (tagSet.stream().anyMatch(tag -> tag.trim().equalsIgnoreCase(NO_TAG_TOKEN))) {
      throw new IllegalArgumentException("The tag '" + NO_TAG_TOKEN + "' is invalid since it's a keyword.");
    }
    return tagSet;
  }

  public void setTags(String tags) {
    this.tags = tags;
  }

  @Override
  public boolean isIgnored() {
    return ignore;
  }

  public void setIgnore(boolean ignore) {
    this.ignore = ignore;
  }

  public AbstractMetadataScope getMetadataScope() {
    return metadataScope;
  }

  public void setMetadataScope(AbstractMetadataScope metadataScope) {
    this.metadataScope = metadataScope;
  }

  public void setValidation(MessageProcessorChain validation) {
    this.validation = validation;
  }

  public MessageProcessorChain getValidation() {
    return validation;
  }

  @Override
  public void stop() throws MuleException {
    stopIfNeeded(metadataScope);
    stopIfNeeded(validation);
  }

  @Override
  public void dispose() {
    disposeIfNeeded(metadataScope, LOGGER);
    disposeIfNeeded(validation, LOGGER);
  }

  @Override
  public void start() throws MuleException {
    startIfNeeded(metadataScope);
    startIfNeeded(validation);
  }

  @Override
  public void initialise() throws InitialisationException {
    initialiseIfNeeded(metadataScope, muleContext);
    initialiseIfNeeded(validation, muleContext);
  }
}
