/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.mtf.tools.internal.tooling.metadata;

import static org.mule.runtime.api.metadata.MetadataService.NON_LAZY_METADATA_SERVICE_KEY;
import static org.mule.runtime.core.api.lifecycle.LifecycleUtils.disposeIfNeeded;
import static org.mule.runtime.core.api.lifecycle.LifecycleUtils.initialiseIfNeeded;
import static org.slf4j.LoggerFactory.getLogger;

import javax.inject.Inject;
import javax.inject.Named;

import org.mule.metadata.api.TypeWriter;
import org.mule.metadata.persistence.api.JsonMetadataTypeWriterFactory;
import org.mule.munit.mtf.tools.internal.tooling.metadata.exception.MetadataTestException;
import org.mule.munit.mtf.tools.internal.tooling.metadata.exception.UnexpectedMetadataFailureException;
import org.mule.runtime.api.component.Component;
import org.mule.runtime.api.component.location.Location;
import org.mule.runtime.api.exception.MuleException;
import org.mule.runtime.api.lifecycle.InitialisationException;
import org.mule.runtime.api.lifecycle.Lifecycle;
import org.mule.runtime.api.message.Message;
import org.mule.runtime.api.metadata.MetadataService;
import org.mule.runtime.api.metadata.resolving.MetadataComponent;
import org.mule.runtime.api.metadata.resolving.MetadataFailure;
import org.mule.runtime.core.api.event.CoreEvent;
import org.mule.runtime.core.privileged.PrivilegedMuleContext;
import org.slf4j.Logger;

import java.util.List;

/**
 * Metadata Scope that calculates metadata over a component
 *
 * @author Mulesoft Inc.
 * @since 1.0.0
 */
public abstract class AbstractMetadataScope implements Lifecycle {

  @Inject
  protected PrivilegedMuleContext muleContext;

  private static Logger LOGGER = getLogger(AbstractMetadataScope.class);

  @Named(NON_LAZY_METADATA_SERVICE_KEY)
  @Inject
  protected MetadataService metadataService;

  protected TypeWriter metadataTypeWriter;

  protected Component component;

  public void setComponent(Component component) {
    this.component = component;
  }

  public Component getComponent() {
    return component;
  }

  public abstract Message getMetadata() throws MetadataTestException;

  @Override
  public void stop() throws MuleException {}

  @Override
  public void dispose() {
    disposeIfNeeded(component, LOGGER);
  }

  @Override
  public void start() throws MuleException {}

  @Override
  public void initialise() throws InitialisationException {
    initialiseIfNeeded(component, muleContext);
    this.metadataTypeWriter = JsonMetadataTypeWriterFactory.create();
  }

  /**
   * Returns an event with the metadata calculated over the {@link AbstractMetadataScope#component}
   *
   * @param incomingEvent incoming event to the scope
   * @return metadata information about the component
   * @throws MetadataTestException a failure occurred during the metadata calculation
   */
  public CoreEvent process(CoreEvent incomingEvent) throws MetadataTestException {
    return CoreEvent.builder(incomingEvent).message(getMetadata()).build();
  }

  /**
   * Gets the {@link Location} of the given {@link Component}
   * 
   * @return location of the component
   */
  protected Location getComponentLocation() {
    return Location.builderFromStringRepresentation(component.getLocation().getLocation()).build();
  }

  /**
   * Returns the {@link MetadataTestException} to throw if the failures contain the expected {@link MetadataComponent}. If the
   * failures are caused by other components, an {@link IllegalArgumentException} is thrown.
   * 
   * @param failures List of failures returned by the {@link org.mule.runtime.api.metadata.resolving.MetadataResult}
   * @param expectedComponent Expected component that failed
   * @return Metadata Exception with the proper {@link MetadataFailure}
   */
  protected MetadataTestException metadataTestException(List<MetadataFailure> failures, MetadataComponent expectedComponent) {
    return new MetadataTestException(failures.stream()
        .filter(metadataFailure -> metadataFailure.getFailingComponent().equals(expectedComponent))
        .findAny()
        .orElseThrow(() -> new UnexpectedMetadataFailureException(expectedComponent, failures)));
  }
}
