/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.mtf.tools.internal.tooling.metadata.processors;

import static java.util.Arrays.asList;
import static java.util.Collections.singletonList;
import static org.apache.commons.lang3.StringUtils.EMPTY;
import static org.skyscreamer.jsonassert.JSONCompareMode.STRICT;

import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.nio.charset.Charset;
import java.util.List;
import java.util.Optional;
import java.util.function.Consumer;

import javax.inject.Inject;
import javax.xml.namespace.QName;

import org.apache.commons.io.IOUtils;
import org.json.JSONArray;
import org.json.JSONObject;
import org.mule.metadata.api.ClassTypeLoader;
import org.mule.metadata.api.TypeWriter;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.json.api.JsonTypeLoader;
import org.mule.metadata.persistence.api.JsonMetadataTypeWriterFactory;
import org.mule.metadata.raml.api.JsonRamlTypeLoader;
import org.mule.metadata.xml.api.SchemaCollector;
import org.mule.metadata.xml.api.XmlTypeLoader;
import org.mule.metadata.xml.api.utils.XmlSchemaUtils;
import org.mule.runtime.api.exception.MuleException;
import org.mule.runtime.api.lifecycle.InitialisationException;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.api.streaming.bytes.CursorStreamProvider;
import org.mule.runtime.core.api.el.ExpressionManager;
import org.mule.runtime.core.api.event.CoreEvent;
import org.mule.runtime.core.privileged.processor.simple.SimpleMessageProcessor;
import org.mule.runtime.extension.api.declaration.type.ExtensionsTypeLoaderFactory;
import org.skyscreamer.jsonassert.JSONAssert;
import org.skyscreamer.jsonassert.JSONParser;

/**
 * Processor that asserts the metadata type against an expected one
 * 
 * @since 1.0.0
 * @author Mulesoft Inc.
 */
public class AssertTypeProcessor extends SimpleMessageProcessor {

  private static final List<String> EXPECTED_METADATA_TYPE_ATTRIBUTES = asList("fromClass", "fromSchema");
  private static final List<String> NON_STRUCTURE_FIELDS = asList("annotations", "format");

  @Inject
  protected ExpressionManager expressionManager;

  private TypeWriter jsonMetadataTypeWriter;
  private ClassTypeLoader classTypeLoader;

  private String actual;
  private String fromClass;
  private String fromSchema;

  @Override
  public CoreEvent process(CoreEvent event) throws MuleException {
    assertMetadataTypeJsons(evaluateActual(event), getExpected());
    return event;
  }

  private String getExpected() {
    if (fromClass != null) {
      return loadFromClass();
    } else if (fromSchema != null) {
      return loadFromSchema();
    }
    throw new IllegalArgumentException("No metadata type to compare was defined. Expected the processor to define one of: "
        + EXPECTED_METADATA_TYPE_ATTRIBUTES);
  }

  @Override
  public void initialise() throws InitialisationException {
    this.jsonMetadataTypeWriter = JsonMetadataTypeWriterFactory.create();
    this.classTypeLoader = ExtensionsTypeLoaderFactory.getDefault().createTypeLoader(getClassLoader());
  }

  public void setActual(String actual) {
    this.actual = actual;
  }

  public void setFromClass(String fromClass) {
    this.fromClass = fromClass;
  }

  public void setFromSchema(String fromSchema) {
    this.fromSchema = fromSchema;
  }

  private String evaluateActual(CoreEvent event) {
    TypedValue content = actual == null ? event.getMessage().getPayload() : expressionManager.evaluate(actual, event);
    Object value = content.getValue();
    try {
      Charset encoding = content.getDataType().getMediaType().getCharset().orElse(Charset.defaultCharset());
      if (value instanceof CursorStreamProvider) {
        return IOUtils.toString(((CursorStreamProvider) value).openCursor(), encoding);
      } else if (value instanceof InputStream) {
        return IOUtils.toString((InputStream) value, encoding);
      } else if (value instanceof byte[]) {
        return new String((byte[]) value, encoding);
      } else if (value instanceof String) {
        return (String) value;
      } else {
        throw new IllegalArgumentException("Parameter 'actual' can not be coerced to a String, was of type: "
            + (value == null ? "null" : value.getClass().getCanonicalName()));
      }
    } catch (IOException ioException) {
      throw new IllegalArgumentException("Parameter 'actual' could not be coerced to a String", ioException);
    }
  }

  private void assertMetadataTypeJsons(String actual, String expected) {
    JSONObject expectedObject = createJsonObject(expected);
    JSONObject actualObject = createJsonObject(actual);
    try {
      JSONAssert.assertEquals(expectedObject, actualObject, STRICT);
    } catch (AssertionError e) {
      throw new AssertionError(e.getMessage());
    }
  }

  private String loadFromClass() {
    try {
      Class clazz = getClassLoader().loadClass(fromClass);
      MetadataType metadataType = classTypeLoader.load(clazz);
      return writeMetadataType(metadataType);
    } catch (ClassNotFoundException e) {
      throw new IllegalArgumentException("Unable to load class " + fromClass, e);
    }
  }

  private String writeMetadataType(MetadataType metadataType) {
    return jsonMetadataTypeWriter.toString(metadataType);
  }

  private String loadFromSchema() {
    URL schemaUrl = getSchemaUrl();
    return loadMetadataType(schemaUrl).map(this::writeMetadataType)
        .orElseThrow(() -> new IllegalArgumentException("Unable to obtain a metadata type from file " + fromSchema));
  }

  private URL getSchemaUrl() {
    URL schemaResource = getClassLoader().getResource(fromSchema);
    if (schemaResource == null) {
      throw new IllegalArgumentException("Unable to locate file " + fromSchema + " as a resource");
    }
    return schemaResource;
  }

  private Optional<MetadataType> loadMetadataType(URL schemaUrl) {
    try {
      String schemaContent = IOUtils.toString(schemaUrl.openStream(), Charset.defaultCharset());
      if (fromSchema.endsWith(".xsd")) {
        return loadXmlSchema(schemaContent);
      } else if (fromSchema.endsWith(".json")) {
        return loadFromJsonSchema(schemaContent);
      } else if (fromSchema.endsWith(".raml")) {
        return loadFromRaml(schemaContent);
      } else {
        throw new IllegalArgumentException("File extension not recognized as a schema file");
      }
    } catch (IOException e) {
      throw new IllegalArgumentException("Unable to read " + fromSchema, e);
    }
  }

  private Optional<MetadataType> loadFromJsonSchema(String schemaContent) {
    return new JsonTypeLoader(schemaContent).load(EMPTY);
  }

  private Optional<MetadataType> loadXmlSchema(String schemaContent) {
    QName rootElementName = XmlSchemaUtils.getXmlSchemaRootElementName(singletonList(schemaContent), EMPTY)
        .orElseThrow(() -> new IllegalArgumentException("Provided schema " + fromSchema + " does not have a root element"));

    SchemaCollector schemaCollector = SchemaCollector.getInstance().addSchema(rootElementName.getLocalPart(), schemaContent);
    return new XmlTypeLoader(schemaCollector).load(rootElementName, null);
  }

  private Optional<MetadataType> loadFromRaml(String schemaContent) {
    return new JsonRamlTypeLoader(schemaContent, fromSchema).load(EMPTY);
  }

  private ClassLoader getClassLoader() {
    return Thread.currentThread().getContextClassLoader();
  }

  private JSONObject createJsonObject(String jsonString) {
    Object json = JSONParser.parseJSON(jsonString);
    if (json instanceof JSONObject) {
      return actionOverObject((JSONObject) json, removeNonComparableFields());
    } else {
      throw new IllegalArgumentException("Assert type can only compare against a Json Object");
    }
  }

  /**
   * Removes the unnecessary fields from the JSONObject so it is not taken into account when comparing two metadata types
   */
  private Consumer<JSONObject> removeNonComparableFields() {
    return jsonObject -> NON_STRUCTURE_FIELDS.forEach(jsonObject::remove);
  }

  private JSONObject actionOverObject(JSONObject jsonObject, Consumer<JSONObject> action) {
    action.accept(jsonObject);
    jsonObject.keys().forEachRemaining(key -> {
      Object field = jsonObject.get(key);
      if (field instanceof JSONObject) {
        actionOverObject((JSONObject) field, action);
      } else if (field instanceof JSONArray) {
        for (Object item : ((JSONArray) field)) {
          if (item instanceof JSONObject) {
            actionOverObject((JSONObject) item, action);
          }
        }
      }
    });
    return jsonObject;
  }

}
