/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.runner.spring.config.document;

import org.mule.config.spring.parsers.DefaultXmlMetadataAnnotations;
import org.mule.config.spring.parsers.XmlMetadataAnnotations;
import org.mule.util.SystemUtils;
import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.w3c.dom.UserDataHandler;
import org.xml.sax.Attributes;
import org.xml.sax.Locator;
import org.xml.sax.SAXException;
import org.xml.sax.helpers.DefaultHandler;

import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Stack;

import static org.mule.config.spring.parsers.XmlMetadataAnnotations.METADATA_ANNOTATIONS_KEY;
import static org.mule.munit.runner.spring.config.document.MunitMetadaAnnotations.MUNIT_METADATA_ANNOTATIONS_KEY;

/**
 * This is a SAX context handler used to add userdata to the node
 */
public class XmlMetadataAnnotator extends DefaultHandler {
    private static final UserDataHandler NULL_DATA_HANDLER = new UserDataHandler() {
        @Override
        public void handle(short operation, String key, Object data, Node src, Node dst) {
            // Nothing to do.
        }
    };

    private static final UserDataHandler COPY_METADATA_ANNOTATIONS_DATA_HANDLER = new UserDataHandler() {
        @Override
        public void handle(short operation, String key, Object data, Node src, Node dst) {
            if (operation == NODE_IMPORTED || operation == NODE_CLONED) {
                dst.setUserData(key, src.getUserData(key),this);
            }
        }
    };

    private Locator locator;
    private DomWalkerElement walker;
    private Stack<XmlMetadataAnnotations> annotationsStack = new Stack<>();

    public XmlMetadataAnnotator(Document doc) {
        this.walker = new DomWalkerElement(doc.getDocumentElement());
    }

    @Override
    public void setDocumentLocator(Locator locator) {
        super.setDocumentLocator(locator);
        this.locator = locator;
    }

    @Override
    public void startElement(String uri, String localName, String qName, Attributes atts) throws SAXException {
        walker = walker.walkIn();

        DefaultXmlMetadataAnnotations metadataBuilder = new DefaultXmlMetadataAnnotations();
        metadataBuilder.setLineNumber(locator.getLineNumber());
        Map<String, String> attsMap = new LinkedHashMap<>();
        for (int i = 0; i < atts.getLength(); i++) {
            attsMap.put(atts.getQName(i), atts.getValue(i));
        }
        metadataBuilder.appendElementStart(qName, attsMap);
        annotationsStack.push(metadataBuilder);
    }

    @Override
    public void characters(char[] ch, int start, int length) throws SAXException {
        annotationsStack.peek().appendElementBody(new String(ch, start, length).trim());
    }

    @Override
    public void endElement(String uri, String localName, String qName) throws SAXException {
        XmlMetadataAnnotations metadataAnnotations = annotationsStack.pop();
        metadataAnnotations.appendElementEnd(qName);

        if (!annotationsStack.isEmpty()) {
            annotationsStack.peek().appendElementBody(SystemUtils.LINE_SEPARATOR + metadataAnnotations.getElementString() + SystemUtils.LINE_SEPARATOR);
        }

        walker.getParentNode().setUserData(METADATA_ANNOTATIONS_KEY, metadataAnnotations, COPY_METADATA_ANNOTATIONS_DATA_HANDLER);

        MunitMetadaAnnotations munitMetadaAnnotations = buildMuniMetadataAnnotation(metadataAnnotations.getLineNumber());
        walker.getParentNode().setUserData(MUNIT_METADATA_ANNOTATIONS_KEY, munitMetadaAnnotations, COPY_METADATA_ANNOTATIONS_DATA_HANDLER);

        walker = walker.walkOut();
    }

    private MunitMetadaAnnotations buildMuniMetadataAnnotation(Integer lineNumber) {
        MunitMetadaAnnotations munitMetadaAnnotations = new MunitMetadaAnnotations();
        munitMetadaAnnotations.putAnnotation(MunitMetadaAnnotations.LINE_NUMBER, lineNumber);

        return munitMetadaAnnotations;
    }
} 
