/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.common.processors;

import net.sf.cglib.asm.Type;
import net.sf.cglib.core.Signature;
import net.sf.cglib.proxy.MethodProxy;
import org.mule.api.MuleEvent;
import org.mule.processor.AbstractInterceptingMessageProcessorBase;

import java.lang.reflect.Method;

/**
 * The goal of this class is to execute the processNext method of an AbstractInterceptingMessageProcessorBase.
 * Providing the messageProcessor provided is indeed an intercepting message processor.
 */
public class InterceptingMessageProcessorHandler {

    private static final String PROCESS_NEXT_METHOD_NAME = "processNext";
    private static final Class PROCESS_NEXT_RETURN_TYPE = MuleEvent.class;
    private static final Class PROCESS_NEXT_ARGUMENT_TYPE = MuleEvent.class;

    private Object messageProcessor;

    public InterceptingMessageProcessorHandler(Object messageProcessor) {
        this.messageProcessor = messageProcessor;
    }

    /**
     * Validate if the message processor should be handle by this class
     *
     * @return true is the messageprocessors extends AbstractInterceptingMessageProcessorBase
     */
    public boolean shouldHandle() {
        if (AbstractInterceptingMessageProcessorBase.class.isAssignableFrom(messageProcessor.getClass())) {
            return true;
        }
        return false;
    }

    /**
     * Handles the call to invokeProcessNext method in the provided message processor.
     *
     * @param event the event to process
     * @return the event result of the process if the message processor provided is an intercepting. If not it returns the same event
     * @throws Throwable
     */
    public Object invokeProcessNext(MuleEvent event) throws Throwable {
        if (!shouldHandle()) {
            return event;
        }


        MethodProxy methodProxy = buildMethodProxy();
        if (methodProxy != null) {
            Object[] args = {event};

            return methodProxy.invokeSuper(messageProcessor, args);
        } else {
            Method method = AbstractInterceptingMessageProcessorBase.class.getDeclaredMethod(PROCESS_NEXT_METHOD_NAME, PROCESS_NEXT_ARGUMENT_TYPE);
            method.setAccessible(true);

            return method.invoke(messageProcessor, event);
        }
    }

    protected MethodProxy buildMethodProxy() {
        Type[] argumentTypes = {Type.getType(PROCESS_NEXT_ARGUMENT_TYPE)};
        Signature signature = new Signature(PROCESS_NEXT_METHOD_NAME, Type.getType(PROCESS_NEXT_RETURN_TYPE), argumentTypes);

        try {
            return MethodProxy.find(messageProcessor.getClass(), signature);
        } catch (IllegalArgumentException e) {
            return null;
        }
    }
}
