/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.runner.spring.config;

import org.mule.api.config.MuleProperties;
import org.mule.api.transport.Connector;
import org.mule.construct.Flow;
import org.springframework.beans.BeansException;
import org.springframework.beans.MutablePropertyValues;
import org.springframework.beans.factory.NoSuchBeanDefinitionException;
import org.springframework.beans.factory.config.BeanDefinition;
import org.springframework.beans.factory.config.ConfigurableListableBeanFactory;
import org.springframework.beans.factory.support.AbstractBeanDefinition;
import org.springframework.beans.factory.support.GenericBeanDefinition;

import java.util.ArrayList;
import java.util.List;
import java.util.logging.Logger;

import static org.mule.modules.interceptor.connectors.ConnectorMethodInterceptorFactory.addFactoryDefinitionTo;

/**
 * <p>
 * This class changes the endpoint factory and inject the mock manager
 * </p>
 * <p/>
 * <p>
 * This is a piece part of the endpoint mocking. By overriding the endpoint factory we can mock all the outbound/inbound
 * endpoints of a mule application
 * </p>
 *
 * @author Mulesoft Inc.
 * @since 3.4.0
 */
public class MunitApplicationContextPostProcessor {

    private static Logger logger = Logger.getLogger("Bean definition Processor");

    /**
     * <p>
     * Defines if the inbounds must be mocked or not. This is pure Munit configuration
     * </p>
     */
    protected boolean mockInbounds = true;

    /**
     * <p>
     * Defines if the app connectors for outbound/inbound endpoints have to be mocked. If they are then all
     * outbound endpoints/inbound endpoints must be mocked.
     * </p>
     */
    protected boolean mockConnectors = true;

    /**
     * <p>
     * List of flows which we don't want to mock the inbound message sources
     * </p>
     */
    protected List<String> mockingExcludedFlows = new ArrayList<String>();


    public void setMockInbounds(boolean mockInbounds) {
        this.mockInbounds = mockInbounds;
    }

    public void setMockingExcludedFlows(List<String> mockingExcludedFlows) {
        this.mockingExcludedFlows = mockingExcludedFlows;
    }

    public boolean isMockInbounds() {
        return mockInbounds;
    }

    public boolean isMockConnectors() {
        return mockConnectors;
    }

    public void setMockConnectors(boolean mockConnectors) {
        this.mockConnectors = mockConnectors;
    }

    /**
     * <p>
     * Implementation of the BeanFactoryPostProcessor. It removes the message sources of all the flows except
     * for the ones specified in mockingExcludedFlows. Only if mockInbounds is true.
     * </p>
     *
     * @param beanFactory <p>
     *                    The spring bean factory
     *                    </p>
     * @throws org.springframework.beans.BeansException <p>
     *                                                  When post processing fails. Never thrown for this implementation
     *                                                  </p>
     */
    public void postProcessBeanFactory(ConfigurableListableBeanFactory beanFactory, Class endpointFactoryClass) throws BeansException {
        if (isMockInbounds() || isMockConnectors()) {
            String[] names = beanFactory.getBeanDefinitionNames();
            String[] beanDefinitionNames = names == null ? new String[0] : names;
            for (String name : beanDefinitionNames) {
                BeanDefinition beanDefinition = beanFactory.getBeanDefinition(name);
                if (Flow.class.getName().equals(beanDefinition.getBeanClassName())) {
                    if (!mockingExcludedFlows.contains(name)) {
                        beanDefinition.getPropertyValues().removePropertyValue("messageSource");
                    }
                }
            }


        }

        changeEndpointFactory(beanFactory, endpointFactoryClass);

        mockConnectors(beanFactory);

    }

    /**
     * <p>
     * Changes the default EndpointFactory of mule with a Wrapper of it. This wrapper creates mocks of the Outbound
     * Endpoints
     * </p>
     *
     * @param beanFactory <p>
     *                    The spring bean factory
     *                    </p>
     */
    private void changeEndpointFactory(ConfigurableListableBeanFactory beanFactory, Class endpointFactoryClass) {
        GenericBeanDefinition endpointFactory = (GenericBeanDefinition) beanFactory.getBeanDefinition(MuleProperties.OBJECT_MULE_ENDPOINT_FACTORY);

        AbstractBeanDefinition abstractBeanDefinition = endpointFactory.cloneBeanDefinition();

        MutablePropertyValues propertyValues = new MutablePropertyValues();
        propertyValues.add("defaultFactory", abstractBeanDefinition);
        endpointFactory.setPropertyValues(propertyValues);
        endpointFactory.setBeanClassName(endpointFactoryClass.getCanonicalName());
    }

    /**
     * <p>
     * Changes the {@link org.mule.api.transport.Connector} bean definition so they are created as mocks of connectors that do not connect
     * </p>
     * <p/>
     * <p>
     * This action is done only if {@link #isMockConnectors()} is true
     * </p>
     *
     * @param beanFactory <p>
     *                    The bean factory that contains the bean definition
     *                    </p>
     */
    private void mockConnectors(ConfigurableListableBeanFactory beanFactory) {
        if (isMockConnectors()) {
            String[] beanNamesForType = beanFactory.getBeanDefinitionNames();
            if (beanNamesForType != null) {
                for (String beanName : beanNamesForType) {
                    BeanDefinition beanDefinition = beanFactory.getBeanDefinition(beanName);

                    Class beanClass = findBeanDefinitionClass(beanDefinition, beanFactory);
                    if (null != beanClass && Connector.class.isAssignableFrom(beanClass)) {
                        AbstractBeanDefinition rootBeanDefinition = AbstractBeanDefinition.class.cast(beanDefinition);

                        if (beanDefinition.getFactoryMethodName() == null) {
                            addFactoryDefinitionTo(rootBeanDefinition).withConstructorArguments(rootBeanDefinition.getBeanClass());
                        } else {
                            logger.info("The connector " + beanName + " cannot be mocked as it already has a factory method");
                        }
                    }

                }
            }
        }
    }

    /**
     * It searches for the bean definition class.
     * This covers the scenario of of Spring beans declared in terms of a parent Spring bean
     *
     * @param beanDefinition
     * @param beanFactory
     * @return null if the bean class for the bean definition could not be found
     */
    private Class findBeanDefinitionClass(BeanDefinition beanDefinition, ConfigurableListableBeanFactory beanFactory) {
        String beanClassName = beanDefinition.getBeanClassName();
        if (null != beanClassName) {
            try {
                return Class.forName((String) beanClassName);
            } catch (ClassNotFoundException e) {
                return null;
            }
        }

        String parentBeanName = beanDefinition.getParentName();
        if (null != parentBeanName) {
            try {
                BeanDefinition parentBeanDefinition = beanFactory.getBeanDefinition(parentBeanName);
                return findBeanDefinitionClass(parentBeanDefinition, beanFactory);
            } catch (NoSuchBeanDefinitionException e) {
                return null;
            }
        }

        return null;
    }
}
