/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.runner.spring.config;

import org.apache.xerces.parsers.DOMParser;
import org.mule.api.MuleContext;
import org.mule.api.config.ConfigurationException;
import org.mule.config.ConfigResource;
import org.mule.config.spring.OptionalObjectsController;
import org.mule.config.spring.SpringXmlConfigurationBuilder;
import org.mule.module.xml.transformer.*;
import org.mule.module.xml.transformer.jaxb.JAXBMarshallerTransformer;
import org.mule.module.xml.transformer.jaxb.JAXBUnmarshallerTransformer;
import org.mule.munit.runner.spring.config.model.BeanDefinitionGenericBuilder;
import org.mule.munit.runner.spring.config.reader.MunitBeanDefinitionScopper;
import org.mule.transformer.simple.*;
import org.springframework.beans.MutablePropertyValues;
import org.springframework.beans.factory.support.RootBeanDefinition;
import org.springframework.context.ApplicationContext;

import java.util.HashMap;
import java.util.Map;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;
import static org.apache.commons.lang.StringUtils.isNotBlank;

/**
 * <p>
 * Override of the {@link org.mule.config.spring.SpringXmlConfigurationBuilder} for Munit. It just overrides the creation of the
 * {@link org.springframework.context.ApplicationContext} in order to create an {@link org.mule.munit.runner.spring.config.MunitApplicationContext}
 * </p>
 *
 * @author Mulesoft Inc.
 * @since 3.3.2
 */
public class MunitSpringXmlConfigurationBuilder extends SpringXmlConfigurationBuilder {

    private BeanDefinitionGenericBuilder mockingConfiguration;

    private DOMParser munitDomParser;

    private Class endpointFactoryClass;
    private Map<String, Class> beansToRegister;

    private Class munitFactoryPostProcessorClass;
    private String munitFactoryPostProcessorBeanId;


    private MunitSpringXmlConfigurationBuilder(String configResources, BeanDefinitionGenericBuilder configuration) throws ConfigurationException {
        super(configResources);
        this.mockingConfiguration = configuration;
        this.beansToRegister = new HashMap<>();

        this.addDefaultBeansToScoppeOut();
    }

    public static class ConfigurationBuilderBuilder {
        private String configResources;

        private BeanDefinitionGenericBuilder mockingConfiguration;

        private DOMParser munitDomParser;

        private Class endpointFactoryClass;
        private Map<String, Class> beansToRegister;

        private Class munitFactoryPostProcessorClass;
        private String munitFactoryPostProcessorBeanId;

        public ConfigurationBuilderBuilder(String configResources) {
            this.configResources = configResources;
            this.beansToRegister = new HashMap<>();
        }

        public ConfigurationBuilderBuilder withMockingConfiguration(BeanDefinitionGenericBuilder mockingConfiguration) {
//            checkNotNull(mockingConfiguration, "The mocking mockingConfiguration must not be null.");
            this.mockingConfiguration = mockingConfiguration;
            return this;
        }

        public ConfigurationBuilderBuilder withEndpointFactoryClass(Class endpointFactoryClass) {
            checkNotNull(endpointFactoryClass, "The class must not be null.");
            this.endpointFactoryClass = endpointFactoryClass;
            return this;
        }

        public ConfigurationBuilderBuilder withMunitFactoryPostProcessor(String beanId, Class postProcessorClass) {
            checkArgument(isNotBlank(beanId), "The bean ID must not be null nor empty.");
            checkNotNull(postProcessorClass, "The class must not be null.");
            this.munitFactoryPostProcessorBeanId = beanId;
            this.munitFactoryPostProcessorClass = postProcessorClass;
            return this;
        }

        public ConfigurationBuilderBuilder withBeanToRegister(String beanId, Class beanClass) {
            checkArgument(isNotBlank(beanId), "The bean ID must not be null nor empty.");
            checkNotNull(beanClass, "The bean class not be null.");
            this.beansToRegister.put(beanId, beanClass);
            return this;
        }

        public ConfigurationBuilderBuilder withMunitDomParser(DOMParser munitDomParser) {
            checkNotNull(munitDomParser, "The dom parsers must not be null.");
            this.munitDomParser = munitDomParser;
            return this;
        }

        public MunitSpringXmlConfigurationBuilder build() throws ConfigurationException {
            checkNotNull(this.endpointFactoryClass, "MunitSpringXmlConfigurationBuilder build fail. The class must not be null."); // not sure about this;
            checkArgument(isNotBlank(this.munitFactoryPostProcessorBeanId), "MunitSpringXmlConfigurationBuilder build fail. The bean ID must not be null nor empty.");
            checkNotNull(this.munitFactoryPostProcessorClass, "MunitSpringXmlConfigurationBuilder build fail. The munit factory post processor class must not be null.");
            checkNotNull(this.munitDomParser, "MunitSpringXmlConfigurationBuilder build fail. The munit dom parsers must not be null"); // not sure about this;

            MunitSpringXmlConfigurationBuilder builder = new MunitSpringXmlConfigurationBuilder(this.configResources, this.mockingConfiguration);
            builder.endpointFactoryClass = this.endpointFactoryClass;
            builder.munitFactoryPostProcessorBeanId = this.munitFactoryPostProcessorBeanId;
            builder.munitFactoryPostProcessorClass = this.munitFactoryPostProcessorClass;
            builder.beansToRegister.putAll(this.beansToRegister);
            builder.munitDomParser = this.munitDomParser;

            return builder;
        }
    }

    public void addBeanClassToScopeOut(String beanClassCanonicalName) {
        Class beanClass = null;
        try {
            beanClass = Class.forName(beanClassCanonicalName);
        } catch (ClassNotFoundException e) {
            throw new RuntimeException("Fail to register class [" + beanClassCanonicalName + "] to scope out." + e.getMessage());
        }
        this.addBeanClassToScopeOut(beanClass);
    }

    public void addBeanClassToScopeOut(Class beanClass) {
        MunitBeanDefinitionScopper.addBeanClass(beanClass);
    }

    public void clearBeanClassList() {
        MunitBeanDefinitionScopper.clearBeanClassList();
    }

    protected void addDefaultBeansToScoppeOut() {
        clearBeanClassList();
//        addBeanClassToScopeOut(Foreach.class);
        addBeanClassToScopeOut(AbstractXmlTransformer.class);

        // Transformers that are MP
        addBeanClassToScopeOut(ObjectToString.class);
        addBeanClassToScopeOut(ByteArrayToObject.class);
        addBeanClassToScopeOut(ByteArrayToSerializable.class);
        addBeanClassToScopeOut(ObjectToString.class);
        addBeanClassToScopeOut(SerializableToByteArray.class);
        addBeanClassToScopeOut(ObjectToByteArray.class);
        addBeanClassToScopeOut(DomDocumentToXml.class);
        addBeanClassToScopeOut(JAXBMarshallerTransformer.class);
        addBeanClassToScopeOut(ObjectToXml.class);
        addBeanClassToScopeOut(XmlToDomDocument.class);
        addBeanClassToScopeOut(JAXBUnmarshallerTransformer.class);
        addBeanClassToScopeOut(XmlToObject.class);
    }

    protected ApplicationContext doCreateApplicationContext(MuleContext muleContext, ConfigResource[] configResources, OptionalObjectsController optionalObjectsController) {
        MunitApplicationContext munitApplicationContext = new MunitApplicationContext(muleContext, configResources, endpointFactoryClass, munitDomParser, munitFactoryPostProcessorBeanId);

        for (String beanId : beansToRegister.keySet()) {
            Class beanClass = beansToRegister.get(beanId);
            munitApplicationContext.putBeanDefinitionToRegister(beanId, new RootBeanDefinition(beanClass));
        }

        if (mockingConfiguration != null) {

            RootBeanDefinition beanDefinition = new RootBeanDefinition();
            beanDefinition.setBeanClass(munitFactoryPostProcessorClass);

            MutablePropertyValues propertyValues = new MutablePropertyValues();
            for (String attributeName : mockingConfiguration.buildAttributeMap().keySet()) {
                Object attributeValue = mockingConfiguration.buildAttributeMap().get(attributeName);
                propertyValues.add(attributeName, attributeValue);
            }
            beanDefinition.setPropertyValues(propertyValues);

            munitApplicationContext.putBeanDefinitionToRegister(munitFactoryPostProcessorBeanId, beanDefinition);
        }

        return munitApplicationContext;
    }

    public BeanDefinitionGenericBuilder getMockingConfiguration() {
        return mockingConfiguration;
    }

    public DOMParser getMunitDomParser() {
        return munitDomParser;
    }

    public Class getEndpointFactoryClass() {
        return endpointFactoryClass;
    }

    public Map<String, Class> getBeansToRegister() {
        return beansToRegister;
    }

    public Class getMunitFactoryPostProcessorClass() {
        return munitFactoryPostProcessorClass;
    }

    public String getMunitFactoryPostProcessorBeanId() {
        return munitFactoryPostProcessorBeanId;
    }
}
