/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.runner.spring.config.reader;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.mule.api.processor.MessageProcessor;
import org.mule.config.spring.factories.ChoiceRouterFactoryBean;
import org.mule.modules.interceptor.processors.MessageProcessorId;
import org.mule.modules.interceptor.spring.BeanFactoryMethodBuilder;
import org.mule.munit.common.MunitConstants;
import org.mule.munit.runner.interceptor.AbstractMessageProcessorInterceptorFactory;
import org.springframework.beans.factory.FactoryBean;
import org.springframework.beans.factory.config.BeanDefinition;
import org.springframework.beans.factory.config.BeanDefinitionHolder;
import org.springframework.beans.factory.support.AbstractBeanDefinition;
import org.springframework.beans.factory.xml.NamespaceHandler;
import org.springframework.beans.factory.xml.ParserContext;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;

import java.util.HashMap;
import java.util.Map;

public class MunitHandlerWrapper implements NamespaceHandler {
    public static  Integer CONSTRUCTOR_ARG_LIMIT = -1;
    private static final Integer DEFAULT_CONSTRUCTOR_ARGS_LIMIT = 5;

    private NamespaceHandler realHandler;
    private transient Log logger = LogFactory.getLog(this.getClass());

    public MunitHandlerWrapper(NamespaceHandler realHandler) {
        this.realHandler = realHandler;
    }

    public static void setConstructorArgLimit(Integer limit) {
        CONSTRUCTOR_ARG_LIMIT = limit;
    }

    @Override
    public void init() {
    }

    @Override
    public BeanDefinition parse(Element element, ParserContext parserContext) {
        BeanDefinition beanDefinition = realHandler.parse(element, parserContext);
        if (beanDefinition == null || element == null) {
            return beanDefinition;
        }

        try {
            Class<?> beanType = Class.forName(beanDefinition.getBeanClassName());
            if (byPassDueToException(beanType)) {
                return beanDefinition;
            }

            if (shouldEnhanceBeanDefinitionWithMunit(beanDefinition)) {

                String tagName = element.getTagName();
                if (isBeanDefinitionEnhanceableByMunit(beanDefinition, tagName)) {

                    String filename = parserContext.getReaderContext().getResource().getFilename();

                    BeanFactoryMethodBuilder methodBuilder = AbstractMessageProcessorInterceptorFactory.addFactoryDefinitionTo((AbstractBeanDefinition) beanDefinition);
                    MessageProcessorId mpId = new MessageProcessorId(getNameFrom(tagName), getNamespaceFrom(tagName));
                    methodBuilder.withConstructorArguments(beanType, mpId, getAttributes(element), filename, element.getAttribute(MunitConstants.LINE_NUMBER_ELEMENT_ATTRIBUTE));


                    MunitBeanDefinitionScopper.makeBeanDefinitionSingletonIfApplicable(beanDefinition);

                    return beanDefinition;
                } else {
                    logger.warn("Bean Definition for " + tagName + " is not enhanceable by MUnit.");
                }
            }


        } catch (ClassNotFoundException e) {
            return beanDefinition;
        }

        return beanDefinition;
    }

    protected boolean shouldEnhanceBeanDefinitionWithMunit(BeanDefinition beanDefinition) throws ClassNotFoundException {
        Class<?> beanType = Class.forName(beanDefinition.getBeanClassName());

        return isMessageProcessor(beanType)
                && AbstractBeanDefinition.class.isAssignableFrom(beanDefinition.getClass())
                && StringUtils.isEmpty(beanDefinition.getFactoryMethodName());
    }

    protected boolean isBeanDefinitionEnhanceableByMunit(BeanDefinition beanDefinition, String tagName) {
        int constructorArgLimit = CONSTRUCTOR_ARG_LIMIT != -1 ? CONSTRUCTOR_ARG_LIMIT : DEFAULT_CONSTRUCTOR_ARGS_LIMIT;

        return !StringUtils.isEmpty(tagName) && beanDefinition.getConstructorArgumentValues().getArgumentCount() <= constructorArgLimit;
    }

    /**
     * Due to certain code in the ESB core that's unstable we are force to by pass some beans.
     * This will cause verifications to fail and coverage problems.
     * <p/>
     * List of Beans we by pass:
     * - Extensions API beans
     * <p/>
     * SINCE 3.7.0
     * TODO: FIX MU-266
     *
     * @param beanType
     * @return
     */
    private boolean byPassDueToException(Class<?> beanType) {
        // TODO:
        // get list of classes to by pass
        // the classes should be tagged by mule version
        // due to backward compatibility issues the current one should remain as it is
        if ("org.mule.module.extension.internal.config.ConfigurationInstanceProviderFactoryBean".equals(beanType.getCanonicalName())) {
            return true;
        }

        // Avoid wrapping choice as it causes issues with mule liqfecycle since 3.7.x and it shouldn't be mocked either
        if (ChoiceRouterFactoryBean.class.getCanonicalName().equals(beanType.getCanonicalName())) {
            return true;
        }

        // Avoid wrapping this bean as the factory is final and it does not create message processors
        if ("com.mulesoft.module.serialization.kryo.internal.config.KryoObjectSerializerFactoryBean".equals(beanType.getCanonicalName())) {
            return true;
        }


        return false;
    }

    private Map<String, String> getAttributes(Element element) {
        Map<String, String> attrs = new HashMap<String, String>();
        NamedNodeMap attributes = element.getAttributes();
        for (int i = 0; i < attributes.getLength(); i++) {
            Node attr = attributes.item(i);
            if (!MunitConstants.LINE_NUMBER_ELEMENT_ATTRIBUTE.equals(attr.getNodeName())) {
                attrs.put(attr.getNodeName(), attr.getNodeValue());
            }
        }
        return attrs;
    }

    private String getNameFrom(String tagName) {
        String[] splitedName = tagName.split(":");
        if (splitedName.length == 1) {
            return splitedName[0];
        } else if (splitedName.length > 1) {
            return splitedName[1];
        }

        return "";
    }

    private String getNamespaceFrom(String tagName) {
        String[] splitedName = tagName.split(":");
        if (splitedName.length > 1) {
            return splitedName[0];
        }

        return "mule";
    }

    private boolean isMessageProcessor(Class<?> beanType) {
        return MessageProcessor.class.isAssignableFrom(beanType)
                || FactoryBean.class.isAssignableFrom(beanType);
    }

    @Override
    public BeanDefinitionHolder decorate(Node source, BeanDefinitionHolder definition, ParserContext parserContext) {
        return definition;
    }
}
