/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.api.spring.config.parsers.model;


import java.util.HashMap;
import java.util.Map;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;
import static org.apache.commons.lang.StringUtils.isNotBlank;


public class ChildElement implements ParseableElement {

    private Class elementClass;

    private String elementName;
    private String elementPropertyName;
    private Map<String, String> attributePropertyMap;

    private Map<String, ParseableElement> childElements;


    public ChildElement() {
        this.attributePropertyMap = new HashMap<>();
        this.childElements = new HashMap<>();
    }

    public static class ChildElementBuilder {
        protected Class elementClass;

        protected String elementName;
        protected String elementPropertyName;
        protected Map<String, String> attributePropertyMap;

        protected HashMap<String, ParseableElement> childElements;

        public ChildElementBuilder(Class elementClass) {
            checkNotNull(elementClass, "The element class must not be null.");
            this.elementClass = elementClass;

            this.attributePropertyMap = new HashMap<>();
            this.childElements = new HashMap<>();
        }

        public ChildElementBuilder withElementName(String elementName) {
            checkArgument(isNotBlank(elementName), "The element name must not be null nor empty.");
            this.elementName = elementName;
            return this;
        }

        public ChildElementBuilder withElementPropertyName(String elementPropertyName) {
            checkArgument(isNotBlank(elementPropertyName), "The element property name must not be null nor empty.");
            this.elementPropertyName = elementPropertyName;
            return this;
        }

        public ChildElementBuilder withAttribute(String attributeName, String propertyName) {
            checkArgument(isNotBlank(attributeName), "The attribute name must not be null nor empty.");
            checkArgument(isNotBlank(propertyName), "The property name must not be null nor empty.");
            this.attributePropertyMap.put(attributeName, propertyName);
            return this;
        }

        public ChildElementBuilder withChildElement(ParseableElement childElement) {
            checkNotNull(childElement, "The child element must not be null.");
            this.childElements.put(childElement.getElementName(), childElement);
            return this;
        }

        public ChildElement build() {
            checkNotNull(elementClass, "Build Fail. The element class must not be null.");
            checkArgument(isNotBlank(elementName), "Build Fail. The element name must not be null.");
            checkArgument(isNotBlank(elementPropertyName), "Build Fail. The  element propertyName name must not be null.");

            ChildElement childElement = new ChildElement();
            childElement.elementClass = this.elementClass;
            childElement.elementName = this.elementName;
            childElement.elementPropertyName = this.elementPropertyName;
            childElement.attributePropertyMap.putAll(this.attributePropertyMap);

            childElement.childElements.putAll(this.childElements);

            return childElement;
        }
    }


    public String getElementName() {
        return elementName;
    }

    public String getElementPropertyName() {
        return elementPropertyName;
    }

    public Class getElementClass() {
        return elementClass;
    }

    public Map<String, String> getAttributePropertyMap() {
        return attributePropertyMap;
    }

    public Map<String, ParseableElement> getChildElements() {
        return childElements;
    }
}


