/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.runner.spring.config.reader;


import com.google.common.base.Preconditions;
import org.springframework.beans.factory.config.BeanDefinition;
import org.springframework.beans.factory.config.ConfigurableBeanFactory;

import java.util.ArrayList;
import java.util.List;

/**
 * The purpose of this class is to change the scope of the bean definition.
 * If the bean definition send to makeBeanDefinitionSingletonIfApplicable is registered in the bean classes list
 * it will then change the scope to SINGLETON.
 * <p/>
 * Otherwise it'll do nothing to it.
 */
public class MunitBeanDefinitionScopper {
    private static List<Class> beanClasses = new ArrayList<>();

    public static synchronized void addBeanClass(Class beanClass) {
        Preconditions.checkNotNull(beanClass, "The bean class must not be null");
        beanClasses.add(beanClass);
    }

    public static synchronized List<Class> getBeanClassList() {
        List<Class> classes = new ArrayList<>();
        classes.addAll(beanClasses);
        return classes;
    }

    public static synchronized void clearBeanClassList() {
        beanClasses.clear();
    }

    public static void makeBeanDefinitionSingletonIfApplicable(BeanDefinition beanDefinition) throws ClassNotFoundException {
        Preconditions.checkNotNull(beanDefinition, "The bean definition must not be null.");

        Class<?> beanType = Class.forName(beanDefinition.getBeanClassName());
        for (Class beanClass : beanClasses) {
            if (beanClass.isAssignableFrom(beanType)) {
                beanDefinition.setScope(ConfigurableBeanFactory.SCOPE_SINGLETON);
            }
        }
    }


}
