/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */

package org.mule.munit.api.spring.config.parsers;

import org.apache.commons.lang.StringUtils;
import org.mule.config.spring.MuleHierarchicalBeanDefinitionParserDelegate;
import org.mule.config.spring.util.SpringXMLUtils;
import org.mule.munit.api.spring.config.parsers.model.*;
import org.mule.util.TemplateParser;
import org.springframework.beans.MutablePropertyValues;
import org.springframework.beans.PropertyValue;
import org.springframework.beans.factory.config.BeanDefinition;
import org.springframework.beans.factory.config.RuntimeBeanReference;
import org.springframework.beans.factory.support.BeanDefinitionBuilder;
import org.springframework.beans.factory.support.ManagedList;
import org.springframework.beans.factory.support.ManagedMap;
import org.springframework.beans.factory.xml.BeanDefinitionParser;
import org.springframework.beans.factory.xml.ParserContext;
import org.springframework.util.xml.DomUtils;
import org.w3c.dom.Element;

import java.util.List;
import java.util.Map;


public abstract class AbstractDefinitionParser implements BeanDefinitionParser, MunitDefinitionParser {

    private TemplateParser.PatternInfo patternInfo;

    public interface ParseDelegate<T> {
        public T parse(Element element);
    }

    public AbstractDefinitionParser() {
        patternInfo = TemplateParser.createMuleStyleParser().getStyle();
    }

    protected boolean hasAttribute(Element element, String attributeName) {
        String value = element.getAttribute(attributeName);
        if ((value != null) && (!StringUtils.isBlank(value))) {
            return true;
        }
        return false;
    }

    protected void setRef(BeanDefinitionBuilder builder, String propertyName, String ref) {
        if (!isMuleExpression(ref)) {
            builder.addPropertyValue(propertyName, new RuntimeBeanReference(ref));
        } else {
            builder.addPropertyValue(propertyName, ref);
        }
    }

    protected boolean isMuleExpression(String value) {
        if ((!value.startsWith(patternInfo.getPrefix())) && (!value.endsWith(patternInfo.getSuffix()))) {
            return false;
        } else {
            return true;
        }
    }

    protected ManagedList parseList(Element element,
                                    String childElementName,
                                    AbstractDefinitionParser.ParseDelegate parserDelegate) {
        ManagedList managedList = new ManagedList();
        List<Element> childDomElements = DomUtils.getChildElementsByTagName(element, childElementName);
        for (Element childDomElement : childDomElements) {
            if (hasAttribute(childDomElement, "value-ref")) {
                if (!isMuleExpression(childDomElement.getAttribute("value-ref"))) {
                    managedList.add(new RuntimeBeanReference(childDomElement.getAttribute("value-ref")));
                } else {
                    managedList.add(childDomElement.getAttribute("value-ref"));
                }
            } else {
                managedList.add(parserDelegate.parse(childDomElement));
            }
        }
        return managedList;
    }

    protected void parseListAndSetProperty(Element element,
                                           BeanDefinitionBuilder builder,
                                           String fieldName,
                                           String parentElementName,
                                           String childElementName,
                                           AbstractDefinitionParser.ParseDelegate parserDelegate) {
        Element domElement = DomUtils.getChildElementByTagName(element, parentElementName);
        if (domElement != null) {
            if (hasAttribute(domElement, "ref")) {
                setRef(builder, fieldName, domElement.getAttribute("ref"));
            } else {
                ManagedList managedList = parseList(domElement, childElementName, parserDelegate);
                builder.addPropertyValue(fieldName, managedList);
            }
        }
    }


    protected ManagedMap parseMap(Element element,
                                  String childElementName,
                                  AbstractDefinitionParser.ParseDelegate parserDelegate) {
        ManagedMap managedMap = new ManagedMap();
        List<Element> childDomElements = DomUtils.getChildElementsByTagName(element, childElementName);
        if (childDomElements.size() == 0) {
            childDomElements = DomUtils.getChildElements(element);
        }
        for (Element childDomElement : childDomElements) {
            Object key = null;
            if (hasAttribute(childDomElement, "key-ref")) {
                key = new RuntimeBeanReference(childDomElement.getAttribute("key-ref"));
            } else {
                if (hasAttribute(childDomElement, "key")) {
                    key = childDomElement.getAttribute("key");
                } else {
                    key = childDomElement.getTagName();
                }
            }
            if (hasAttribute(childDomElement, "value-ref")) {
                if (!isMuleExpression(childDomElement.getAttribute("value-ref"))) {
                    managedMap.put(key, new RuntimeBeanReference(childDomElement.getAttribute("value-ref")));
                } else {
                    managedMap.put(key, childDomElement.getAttribute("value-ref"));
                }
            }
            if (hasAttribute(childDomElement, "value")) {
                managedMap.put(key, childDomElement.getAttribute("value"));
            } else {
                managedMap.put(key, parserDelegate.parse(childDomElement));
            }
        }
        return managedMap;
    }

    protected void parseMapAndSetProperty(Element element,
                                          BeanDefinitionBuilder builder,
                                          String fieldName,
                                          String parentElementName,
                                          String childElementName,
                                          AbstractDefinitionParser.ParseDelegate parserDelegate) {
        Element domElement = DomUtils.getChildElementByTagName(element, parentElementName);
        if (domElement != null) {
            if (hasAttribute(domElement, "ref")) {
                setRef(builder, fieldName, domElement.getAttribute("ref"));
            } else {
                ManagedMap managedMap = parseMap(domElement, childElementName, parserDelegate);
                builder.addPropertyValue(fieldName, managedMap);
            }
        }
    }

    protected void attachProcessorDefinition(ParserContext parserContext, BeanDefinition definition) {
        MutablePropertyValues propertyValues = parserContext.getContainingBeanDefinition()
                .getPropertyValues();
        if (parserContext.getContainingBeanDefinition()
                .getBeanClassName()
                .equals("org.mule.config.spring.factories.PollingMessageSourceFactoryBean")) {
            propertyValues.addPropertyValue("messageProcessor", definition);
        } else {
            if (parserContext.getContainingBeanDefinition()
                    .getBeanClassName()
                    .equals("org.mule.enricher.MessageEnricher")) {
                propertyValues.addPropertyValue("enrichmentMessageProcessor", definition);
            } else {
                PropertyValue messageProcessors = propertyValues.getPropertyValue("messageProcessors");
                if ((messageProcessors == null) || (messageProcessors.getValue() == null)) {
                    propertyValues.addPropertyValue("messageProcessors", new ManagedList());
                }
                List listMessageProcessors = ((List) propertyValues.getPropertyValue("messageProcessors")
                        .getValue());
                listMessageProcessors.add(definition);
            }
        }
    }


    protected void setNoRecurseOnDefinition(BeanDefinition definition) {
        definition.setAttribute(MuleHierarchicalBeanDefinitionParserDelegate.MULE_NO_RECURSE, Boolean.TRUE);
    }

    protected String generateChildBeanName(Element element) {
        String id = SpringXMLUtils.getNameOrId(element);
        if (StringUtils.isBlank(id)) {
            String parentId = SpringXMLUtils.getNameOrId(((Element) element.getParentNode()));
            return ((("." + parentId) + ":") + element.getLocalName());
        } else {
            return id;
        }
    }


    protected List parseNestedProcessorAsList(Element element, ParserContext parserContext, Class factory) {
        BeanDefinitionBuilder builder = BeanDefinitionBuilder.rootBeanDefinition(factory);
        BeanDefinition beanDefinition = builder.getBeanDefinition();
        parserContext.getRegistry().registerBeanDefinition(generateChildBeanName(element), beanDefinition);
        element.setAttribute("name", generateChildBeanName(element));
//        builder.setSource(parserContext.extractSource(element));
        builder.setScope(BeanDefinition.SCOPE_SINGLETON);
        List list = parserContext.getDelegate().parseListElement(element, builder.getBeanDefinition());
        parserContext.getRegistry().removeBeanDefinition(generateChildBeanName(element));
        return list;
    }

    protected void parseNestedProcessorAsListAndSetProperty(Element element,
                                                            String childElementName,
                                                            ParserContext parserContext,
                                                            Class factory,
                                                            BeanDefinitionBuilder builder,
                                                            String propertyName) {
        Element childDomElement = DomUtils.getChildElementByTagName(element, childElementName);
        if (childDomElement != null) {
            builder.addPropertyValue(propertyName,
                    parseNestedProcessorAsList(childDomElement, parserContext, factory));
        }
    }

    protected void parseElementAttributes(Map<String, String> attributePropetyMap, Element element, BeanDefinitionBuilder builder) {
        for (String attributeName : attributePropetyMap.keySet()) {
            String propertyName = attributePropetyMap.get(attributeName);

            if (hasAttribute(element, attributeName)) {
                String attributeValue = element.getAttribute(attributeName);
                builder.addPropertyValue(propertyName, buildAttributeValue(attributeName, attributeValue));
            }
        }
    }

    protected Object buildAttributeValue(String attributeName, String attributeValue) {
        if (attributeName.endsWith("-ref")) {
            if (attributeValue.startsWith("#[")) {
                return attributeValue;
            } else {
                return new RuntimeBeanReference(attributeValue);
            }
        } else {
            return attributeValue;
        }
    }

    protected void parseChildElementByType(ParseableElement parseableElement, Element element, BeanDefinitionBuilder builder, ParserContext parserContext) {
        if (parseableElement instanceof ChildElement) {
            parseChildElement((ChildElement) parseableElement, element, builder, parserContext);
        }

        if (parseableElement instanceof ListChildElement) {
            parseListChildElement((ListChildElement) parseableElement, element, builder);
        }

        if (parseableElement instanceof MapChildElement) {
            parseMapChildElement((MapChildElement) parseableElement, element, builder);
        }

        if (parseableElement instanceof NestedChildElement) {
            parseNestedChildelement((NestedChildElement) parseableElement, element, builder, parserContext);
        }
    }

    protected void parseChildElement(ChildElement childElement, Element element, BeanDefinitionBuilder builder, ParserContext parserContext) {
        BeanDefinitionBuilder childElementBuilder = BeanDefinitionBuilder.rootBeanDefinition(childElement.getElementClass().getName());

        Element domChildElement = DomUtils.getChildElementByTagName(element, childElement.getElementName());
        if (domChildElement != null) {
            parseElementAttributes(childElement.getAttributePropertyMap(), domChildElement, childElementBuilder);

            for (String elementName : childElement.getChildElements().keySet()) {
                ParseableElement parseableElement = childElement.getChildElements().get(elementName);
                parseChildElementByType(parseableElement, domChildElement, childElementBuilder, parserContext);
            }

            builder.addPropertyValue(childElement.getElementPropertyName(), childElementBuilder.getBeanDefinition());
        }
    }

    protected void parseListChildElement(final ListChildElement listChildElement, Element element, BeanDefinitionBuilder builder) {
        parseListAndSetProperty(element, builder, listChildElement.getElementPropertyName(), listChildElement.getElementName(), listChildElement.getItemName(), new ParseDelegate<BeanDefinition>() {
                    public BeanDefinition parse(Element element) {
                        BeanDefinitionBuilder builder = BeanDefinitionBuilder.rootBeanDefinition(listChildElement.getItemClass());
                        parseElementAttributes(listChildElement.getItemAttributePropertyMap(), element, builder);
                        return builder.getBeanDefinition();
                    }
                }
        );
    }

    protected void parseMapChildElement(final MapChildElement mapChildElement, Element element, BeanDefinitionBuilder builder) {
        parseMapAndSetProperty(element, builder, mapChildElement.getElementPropertyName(), mapChildElement.getElementName(), mapChildElement.getItemName(), new ParseDelegate<String>() {
                    public String parse(Element element) {
                        return element.getTextContent();
                    }

                }
        );
    }

    protected void parseNestedChildelement(final NestedChildElement nestedChildElement, Element element, BeanDefinitionBuilder builder, ParserContext parserContext) {
        String childElementName = nestedChildElement.getElementName();
        String propertyName = nestedChildElement.getElementPropertyName();

        parseNestedProcessorAsListAndSetProperty(element, childElementName, parserContext, nestedChildElement.getElementClass(), builder, propertyName);
    }

    //    protected ManagedSet parseSet(Element element,
//                                  String childElementName,
//                                  AbstractDefinitionParser.ParseDelegate parserDelegate) {
//        ManagedSet managedSet = new ManagedSet();
//        List<Element> childDomElements = DomUtils.getChildElementsByTagName(element, childElementName);
//        for (Element childDomElement : childDomElements) {
//            if (hasAttribute(childDomElement, "value-ref")) {
//                if (!isMuleExpression(childDomElement.getAttribute("value-ref"))) {
//                    managedSet.add(new RuntimeBeanReference(childDomElement.getAttribute("value-ref")));
//                } else {
//                    managedSet.add(childDomElement.getAttribute("value-ref"));
//                }
//            } else {
//                managedSet.add(parserDelegate.parse(childDomElement));
//            }
//        }
//        return managedSet;
//    }
//    protected void parseSetAndSetProperty(Element element,
//                                          BeanDefinitionBuilder builder,
//                                          String fieldName,
//                                          String parentElementName,
//                                          String childElementName,
//                                          AbstractDefinitionParser.ParseDelegate parserDelegate) {
//        Element domElement = DomUtils.getChildElementByTagName(element, parentElementName);
//        if (domElement != null) {
//            if (hasAttribute(domElement, "ref")) {
//                setRef(builder, fieldName, domElement.getAttribute("ref"));
//            } else {
//                ManagedSet managedSet = parseSet(domElement, childElementName, parserDelegate);
//                builder.addPropertyValue(fieldName, managedSet);
//            }
//        }
//    }
//    protected String getAttributeValue(Element element, String attributeName) {
//        if (!StringUtils.isEmpty(element.getAttribute(attributeName))) {
//            return element.getAttribute(attributeName);
//        }
//        return null;
//    }
//    protected BeanDefinition parseNestedProcessor(Element element, ParserContext parserContext, Class factory) {
//        BeanDefinitionBuilder builder = BeanDefinitionBuilder.rootBeanDefinition(factory);
//        BeanDefinition beanDefinition = builder.getBeanDefinition();
//        parserContext.getRegistry().registerBeanDefinition(generateChildBeanName(element), beanDefinition);
//        element.setAttribute("name", generateChildBeanName(element));
////        builder.setSource(parserContext.extractSource(element));
//        builder.setScope(BeanDefinition.SCOPE_SINGLETON);
//        List list = parserContext.getDelegate().parseListElement(element, builder.getBeanDefinition());
//        parserContext.getRegistry().removeBeanDefinition(generateChildBeanName(element));
//        return beanDefinition;
//    }
    //    protected void parseSetWithDefaultAndSetProperty(Element element,
//                                                     BeanDefinitionBuilder builder,
//                                                     String fieldName,
//                                                     String parentElementName,
//                                                     String childElementName,
//                                                     String defaultValue,
//                                                     AbstractMockDefinitionParser.ParseDelegate parserDelegate) {
//        Element domElement = DomUtils.getChildElementByTagName(element, parentElementName);
//        if (domElement != null) {
//            if (hasAttribute(domElement, "ref")) {
//                setRef(builder, fieldName, domElement.getAttribute("ref"));
//            } else {
//                ManagedSet managedSet = parseSet(domElement, childElementName, parserDelegate);
//                builder.addPropertyValue(fieldName, managedSet);
//            }
//        } else {
//            builder.addPropertyValue(fieldName, defaultValue);
//        }
//    }
//    protected void parseMapWithDefaultAndSetProperty(Element element,
//                                                     BeanDefinitionBuilder builder,
//                                                     String fieldName,
//                                                     String parentElementName,
//                                                     String childElementName,
//                                                     String defaultValue,
//                                                     AbstractMockDefinitionParser.ParseDelegate parserDelegate) {
//        Element domElement = DomUtils.getChildElementByTagName(element, parentElementName);
//        if (domElement != null) {
//            if (hasAttribute(domElement, "ref")) {
//                setRef(builder, fieldName, domElement.getAttribute("ref"));
//            } else {
//                ManagedMap managedMap = parseMap(domElement, childElementName, parserDelegate);
//                builder.addPropertyValue(fieldName, managedMap);
//            }
//        } else {
//            builder.addPropertyValue(fieldName, defaultValue);
//        }
//    }
//
//    protected void parseListWithDefaultAndSetProperty(Element element,
//                                                      BeanDefinitionBuilder builder,
//                                                      String fieldName,
//                                                      String parentElementName,
//                                                      String childElementName,
//                                                      String defaultValue,
//                                                      AbstractMockDefinitionParser.ParseDelegate parserDelegate) {
//        Element domElement = DomUtils.getChildElementByTagName(element, parentElementName);
//        if (domElement != null) {
//            if (hasAttribute(domElement, "ref")) {
//                setRef(builder, fieldName, domElement.getAttribute("ref"));
//            } else {
//                ManagedList managedList = parseList(domElement, childElementName, parserDelegate);
//                builder.addPropertyValue(fieldName, managedList);
//            }
//        } else {
//            builder.addPropertyValue(fieldName, defaultValue);
//        }
//    }
//
//    protected void parseConfigRef(Element element, BeanDefinitionBuilder builder) {
//        if (hasAttribute(element, "config-ref")) {
//            String configRef = element.getAttribute("config-ref");
//            if (configRef.startsWith("#[")) {
//                builder.addPropertyValue("moduleObject", configRef);
//            } else {
//                builder.addPropertyValue("moduleObject", new RuntimeBeanReference(configRef));
//            }
//        }
//    }
//    protected void attachSourceDefinition(ParserContext parserContext, BeanDefinition definition) {
//        MutablePropertyValues propertyValues = parserContext.getContainingBeanDefinition()
//                .getPropertyValues();
//        propertyValues.addPropertyValue("messageSource", definition);
//    }
//    protected void parseConfigName(Element element) {
//        if (hasAttribute(element, "name")) {
//            element.setAttribute("name", AutoIdUtils.getUniqueName(element, "mule-bean"));
//        }
//    }
//
//    protected void setInitMethodIfNeeded(BeanDefinitionBuilder builder, Class clazz) {
//        if (Initialisable.class.isAssignableFrom(clazz)) {
//            builder.setInitMethodName(Initialisable.PHASE_NAME);
//        }
//    }
//
//    protected void setDestroyMethodIfNeeded(BeanDefinitionBuilder builder, Class clazz) {
//        if (Disposable.class.isAssignableFrom(clazz)) {
//            builder.setDestroyMethodName(Disposable.PHASE_NAME);
//        }
//    }
//
//    protected void parseProperty(BeanDefinitionBuilder builder, Element element, String propertyName) {
//        parseProperty(builder, element, propertyName, propertyName);
//    }
//
//    protected void parseProperty(BeanDefinitionBuilder builder,
//                                 Element element,
//                                 String attributeName,
//                                 String propertyName) {
//        if (hasAttribute(element, attributeName)) {
//            builder.addPropertyValue(propertyName, element.getAttribute(attributeName));
//        }
//    }
//    protected void parseNestedProcessorAsListAndSetProperty(Element element,
//                                                            ParserContext parserContext,
//                                                            Class factory,
//                                                            BeanDefinitionBuilder builder,
//                                                            String propertyName) {
//        builder.addPropertyValue(propertyName, parseNestedProcessorAsList(element, parserContext, factory));
//    }
//
//    protected void parseNestedProcessorAndSetProperty(Element element,
//                                                      ParserContext parserContext,
//                                                      Class factory,
//                                                      BeanDefinitionBuilder builder,
//                                                      String propertyName) {
//        builder.addPropertyValue(propertyName, parseNestedProcessor(element, parserContext, factory));
//    }
//    protected void parseNestedProcessorAndSetProperty(Element element,
//                                                      String childElementName,
//                                                      ParserContext parserContext,
//                                                      Class factory,
//                                                      BeanDefinitionBuilder builder,
//                                                      String propertyName) {
//        Element childDomElement = DomUtils.getChildElementByTagName(element, childElementName);
//        if (childDomElement != null) {
//            builder.addPropertyValue(propertyName,
//                    parseNestedProcessor(childDomElement, parserContext, factory));
//        }
//    }
//
//    protected void parseRetryPolicyTemplate(String elementName,
//                                            Element element,
//                                            ParserContext parserContext,
//                                            BeanDefinitionBuilder builder,
//                                            BeanDefinition definition) {
//        Element reconnectElement = DomUtils.getChildElementByTagName(element, elementName);
//        if (reconnectElement != null) {
//            parserContext.getRegistry()
//                    .registerBeanDefinition(getAttributeValue(element, "name"), definition);
//            BeanDefinition retryPolicyBeanDefinition = parserContext.getDelegate().parseCustomElement(
//                    reconnectElement);
//            parserContext.getRegistry().removeBeanDefinition(getAttributeValue(element, "name"));
//            builder.addPropertyValue("retryPolicyTemplate", retryPolicyBeanDefinition);
//        }
//    }
//
//    protected boolean parseObjectRef(Element element,
//                                     BeanDefinitionBuilder builder,
//                                     String elementName,
//                                     String propertyName) {
//        Element childElement = DomUtils.getChildElementByTagName(element, elementName);
//        if (childElement != null) {
//            if (hasAttribute(childElement, "ref")) {
//                if (childElement.getAttribute("ref").startsWith("#")) {
//                    builder.addPropertyValue(propertyName, childElement.getAttribute("ref"));
//                } else {
//                    builder.addPropertyValue(propertyName,
//                            (("#[registry:" + childElement.getAttribute("ref")) + "]"));
//                }
//                return true;
//            }
//        }
//        return false;
//    }
//
//    protected boolean parseObjectRefWithDefault(Element element,
//                                                BeanDefinitionBuilder builder,
//                                                String elementName,
//                                                String propertyName,
//                                                String defaultValue) {
//        Element childElement = DomUtils.getChildElementByTagName(element, elementName);
//        if (childElement != null) {
//            if (hasAttribute(childElement, "ref")) {
//                if (childElement.getAttribute("ref").startsWith("#")) {
//                    builder.addPropertyValue(propertyName, childElement.getAttribute("ref"));
//                } else {
//                    builder.addPropertyValue(propertyName,
//                            (("#[registry:" + childElement.getAttribute("ref")) + "]"));
//                }
//                return true;
//            }
//        } else {
//            builder.addPropertyValue(propertyName, defaultValue);
//        }
//        return false;
//    }
//
//    protected boolean parseNoExprObjectRef(Element element,
//                                           BeanDefinitionBuilder builder,
//                                           String elementName,
//                                           String propertyName) {
//        Element childElement = DomUtils.getChildElementByTagName(element, elementName);
//        if (childElement != null) {
//            if (hasAttribute(childElement, "ref")) {
//                if (childElement.getAttribute("ref").startsWith("#")) {
//                    builder.addPropertyValue(propertyName, childElement.getAttribute("ref"));
//                } else {
//                    builder.addPropertyValue(propertyName,
//                            new RuntimeBeanReference(childElement.getAttribute("ref")));
//                }
//                return true;
//            }
//        }
//        return false;
//    }
//
//    protected boolean parseNoExprObjectRefWithDefault(Element element,
//                                                      BeanDefinitionBuilder builder,
//                                                      String elementName,
//                                                      String propertyName,
//                                                      String defaultValue) {
//        Element childElement = DomUtils.getChildElementByTagName(element, elementName);
//        if (childElement != null) {
//            if (hasAttribute(childElement, "ref")) {
//                if (childElement.getAttribute("ref").startsWith("#")) {
//                    builder.addPropertyValue(propertyName, childElement.getAttribute("ref"));
//                } else {
//                    builder.addPropertyValue(propertyName,
//                            new RuntimeBeanReference(childElement.getAttribute("ref")));
//                }
//                return true;
//            }
//        } else {
//            builder.addPropertyValue(propertyName, defaultValue);
//        }
//        return false;
//    }
//    protected void parsePropertyRef(BeanDefinitionBuilder builder,
//                                    Element element,
//                                    String attributeName,
//                                    String propertyName) {
//        if (hasAttribute(element, attributeName)) {
//            builder.addPropertyValue(propertyName,
//                    new RuntimeBeanReference(element.getAttribute(attributeName)));
//        }
//    }
//
//    protected void parsePropertyRef(BeanDefinitionBuilder builder, Element element, String propertyName) {
//        parsePropertyRef(builder, element, propertyName, propertyName);
//    }
//
//    protected void parseTextProperty(BeanDefinitionBuilder builder,
//                                     Element element,
//                                     String elementName,
//                                     String propertyName) {
//        Element childElement = DomUtils.getChildElementByTagName(element, elementName);
//        if (childElement != null) {
//            builder.addPropertyValue(propertyName, childElement.getTextContent());
//        }
//    }


}
