/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.runner.spring.config.document;

import org.apache.commons.io.IOUtils;
import org.apache.xerces.parsers.DOMParser;
import org.springframework.beans.factory.xml.DefaultDocumentLoader;
import org.springframework.beans.factory.xml.DocumentLoader;
import org.w3c.dom.Document;
import org.xml.sax.*;

import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;

import static com.google.common.base.Preconditions.checkNotNull;

/**
 * <p>
 * We change the document loader in order to get the line numbers as element attribute.
 * </p>
 * <p/>
 * This loaded uses SAX to parse the XML and instead of add attributes to the node it adds all data to the userdata section of the node
 *
 * @author Mulesoft Inc.
 * @since 3.8
 */
public class MunitAnnotatedDocumentLoader implements DocumentLoader {
    private DOMParser munitDomParser;
    private DocumentLoader defaultLoader = new DefaultDocumentLoader();

    public MunitAnnotatedDocumentLoader(DOMParser munitDomParser) {
        checkNotNull(munitDomParser, "The dom parsers must not be null.");
        this.munitDomParser = munitDomParser;
    }

    /**
     * Load the {@link Document} at the supplied {@link InputSource} using the standard JAXP-configured XML parser.
     */
    @Override
    public Document loadDocument(InputSource inputSource, EntityResolver entityResolver,
                                 ErrorHandler errorHandler, int validationMode, boolean namespaceAware) throws Exception {
        ByteArrayOutputStream output = new ByteArrayOutputStream();
        IOUtils.copy(inputSource.getByteStream(), output);

        InputSource defaultInputSource = new InputSource(new ByteArrayInputStream(output.toByteArray()));
        InputSource enrichInputSource = new InputSource(new ByteArrayInputStream(output.toByteArray()));

        Document doc = defaultLoader.loadDocument(defaultInputSource, entityResolver, errorHandler, validationMode, namespaceAware);

        createSaxAnnotator(doc).parse(enrichInputSource);

        return doc;
    }

    protected XMLReader createSaxAnnotator(Document doc) throws ParserConfigurationException, SAXException {
        SAXParserFactory saxParserFactory = SAXParserFactory.newInstance();
        SAXParser saxParser = saxParserFactory.newSAXParser();
        XMLReader documentReader = saxParser.getXMLReader();
        documentReader.setContentHandler(new XmlMetadataAnnotator(doc));
        return documentReader;
    }

}
