/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.runner.spring.config;

import org.apache.xerces.parsers.DOMParser;
import org.mule.api.MuleContext;
import org.mule.config.ConfigResource;
import org.mule.config.spring.MissingParserProblemReporter;
import org.mule.config.spring.MuleArtifactContext;
import org.mule.munit.runner.interceptor.AbstractMessageProcessorInterceptorFactory;
import org.mule.munit.runner.spring.config.document.MunitAnnotatedDocumentLoader;
import org.mule.munit.runner.spring.config.reader.MunitBeanDefinitionDocumentReader;
import org.springframework.beans.BeansException;
import org.springframework.beans.MutablePropertyValues;
import org.springframework.beans.factory.config.BeanDefinition;
import org.springframework.beans.factory.config.ConfigurableListableBeanFactory;
import org.springframework.beans.factory.support.DefaultListableBeanFactory;
import org.springframework.beans.factory.xml.XmlBeanDefinitionReader;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;
import static org.apache.commons.lang.StringUtils.isNotBlank;

/**
 * <p>
 * The {@link MunitApplicationContext} that represents an app running with Munit.
 * </p>
 * <p/>
 * <p>
 * The main difference between {@link MunitApplicationContext} and {@link org.mule.config.spring.MuleArtifactContext} is that
 * it changes the bean definition reader in order to make the Mule stacktrace work and also registers the Bean definition
 * of the MockingConfiguration
 * </p>
 *
 * @author Mulesoft Inc.
 * @since 3.3.2
 */
public class MunitApplicationContext extends MuleArtifactContext {

    private Map<String, BeanDefinition> beanDefinitionsToRegister;

    private Class endpointFactoryClass;

    private DOMParser munitDomParser;

    private String munitFactoryPostProcessorId;

    public MunitApplicationContext(MuleContext muleContext, ConfigResource[] configResources, Class endpointFactoryClass, DOMParser munitDomParser, String munitFactoryPostProcessorId) throws BeansException {
        super(muleContext, configResources);

        this.beanDefinitionsToRegister = new HashMap<>();

        checkNotNull(endpointFactoryClass, "The endpoint Factory class must not be null.");
        this.endpointFactoryClass = endpointFactoryClass;

        checkNotNull(munitDomParser, "The dom parsers must not be null.");
        this.munitDomParser = munitDomParser;

        checkArgument(isNotBlank(munitFactoryPostProcessorId), "The munit factory post processor id must not be null nor empty.");
        this.munitFactoryPostProcessorId = munitFactoryPostProcessorId;
    }

    public void putBeanDefinitionToRegister(String beanId, BeanDefinition beanDefinition) {
        checkArgument(isNotBlank(beanId), "The bean ID must not be null nor empty.");
        checkNotNull(beanDefinition, "The bean definition not be null.");

        this.beanDefinitionsToRegister.put(beanId, beanDefinition);
    }


    @Override
    protected void loadBeanDefinitions(DefaultListableBeanFactory beanFactory) throws IOException {
        checkNotNull(beanFactory, "The bean factory must not be null.");

        XmlBeanDefinitionReader beanDefinitionReader = getMunitXmlBeanDefinitionReader(beanFactory);

        beanDefinitionReader.setDocumentReaderClass(MunitBeanDefinitionDocumentReader.class);

        this.registerBeanDefinitions(beanFactory);

        //Add error reporting
        beanDefinitionReader.setProblemReporter(new MissingParserProblemReporter());

        // TODO THIS SOUNDS LIKE AN AWFUL HACK
        //beanFactory.getBean(MunitMessageProcessorInterceptorFactory.ID);
        this.triggerBeanCreation(beanFactory);

        this.communicateMuleContextToParsers(beanDefinitionReader);
    }

    protected void triggerBeanCreation(DefaultListableBeanFactory beanFactory) {
        for (String beanId : beanDefinitionsToRegister.keySet()) {
            beanFactory.getBean(beanId);
        }
    }

    protected void registerBeanDefinitions(DefaultListableBeanFactory beanFactory) {
        for (String beanId : beanDefinitionsToRegister.keySet()) {
            BeanDefinition beanDefinition = beanDefinitionsToRegister.get(beanId);
            beanFactory.registerBeanDefinition(beanId, beanDefinition);
        }
    }

    protected XmlBeanDefinitionReader getMunitXmlBeanDefinitionReader(DefaultListableBeanFactory beanFactory) {
        XmlBeanDefinitionReader beanDefinitionReader = (XmlBeanDefinitionReader) createBeanDefinitionReader(beanFactory);

        beanDefinitionReader.setDocumentLoader(new MunitAnnotatedDocumentLoader(this.munitDomParser));

        return beanDefinitionReader;
    }

    protected void communicateMuleContextToParsers(XmlBeanDefinitionReader beanDefinitionReader) {
        getCurrentMuleContext().set(this.getMuleContext());

        beanDefinitionReader.loadBeanDefinitions(getConfigResources());

        getCurrentMuleContext().remove();
    }

    @Override
    protected void prepareBeanFactory(ConfigurableListableBeanFactory beanFactory) {
        super.prepareBeanFactory(beanFactory);

        BeanDefinition beanDefinition = beanFactory.getBeanDefinition(munitFactoryPostProcessorId);
        MutablePropertyValues propertyValues = beanDefinition.getPropertyValues();
        MunitApplicationContextPostProcessor postProcessor = new MunitApplicationContextPostProcessor();

        //TODO: review this when working on runner the casting here is not pritty
        // postProcessor.setMockConnectors((Boolean) propertyValues.getPropertyValue("mockConnectors").getValue());
        // postProcessor.setMockInbounds((Boolean) propertyValues.getPropertyValue("mockInbounds").getValue());
        // TODO: this is because we are not parsing exclude flows anymore is only used in the runner -also the schema doesn't allows it. We should remove it as is
        // postProcessor.setMockingExcludedFlows((List) propertyValues.getPropertyValue("mockingExcludedFlows").getValue());

        // TODO this may need to change I don't like the hardcoded property names
        String mockConnectorsPropName = "mockConnectors";
        String mockInboundsPropName = "mockInbounds";
        String mockingExcludedFlows = "mockingExcludedFlows";

        checkNotNull(propertyValues.getPropertyValue(mockConnectorsPropName), "The property " + mockConnectorsPropName + " is not present in the bean " + munitFactoryPostProcessorId);
        checkNotNull(propertyValues.getPropertyValue(mockInboundsPropName), "The property " + mockInboundsPropName + " is not present in the bean " + munitFactoryPostProcessorId);
        if (propertyValues.getPropertyValue(mockConnectorsPropName).getValue().getClass().isAssignableFrom(Boolean.class)) {
            postProcessor.setMockConnectors((Boolean) propertyValues.getPropertyValue(mockConnectorsPropName).getValue());
        } else {
            postProcessor.setMockConnectors(Boolean.valueOf((String) propertyValues.getPropertyValue(mockConnectorsPropName).getValue()));
        }
        if (propertyValues.getPropertyValue(mockInboundsPropName).getValue().getClass().isAssignableFrom(Boolean.class)) {
            postProcessor.setMockInbounds((Boolean) propertyValues.getPropertyValue(mockInboundsPropName).getValue());
        } else {
            postProcessor.setMockInbounds(Boolean.valueOf((String) propertyValues.getPropertyValue(mockInboundsPropName).getValue()));
        }

        if (null == propertyValues.getPropertyValue(mockingExcludedFlows)) {
            postProcessor.setMockingExcludedFlows(new ArrayList<String>());
        } else {
            if (List.class.isAssignableFrom(propertyValues.getPropertyValue(mockingExcludedFlows).getValue().getClass())) {
                postProcessor.setMockingExcludedFlows((List) propertyValues.getPropertyValue(mockingExcludedFlows).getValue());
            } else {
                postProcessor.setMockingExcludedFlows(new ArrayList<String>());
            }
        }

        postProcessor.postProcessBeanFactory(beanFactory, this.endpointFactoryClass);

    }


    @Override
    public <T> Map<String, T> getBeansOfType(Class<T> type) throws BeansException {
        return getBeansOfType(type, true, true);
    }

    @Override
    public <T> Map<String, T> getBeansOfType(Class<T> type, boolean includeNonSingletons, boolean allowEagerInit) throws BeansException {
        Map<String, T> result = super.getBeansOfType(type, includeNonSingletons, allowEagerInit);

        if (result.isEmpty()) {
            String[] beanDefinitionNames = super.getBeanDefinitionNames();
            for (String beanDefinitionName : beanDefinitionNames) {
                BeanDefinition beanDefinition = super.getBeanFactory().getBeanDefinition(beanDefinitionName);
                if ("create".equals(beanDefinition.getFactoryMethodName()) && AbstractMessageProcessorInterceptorFactory.ID.equals(beanDefinition.getFactoryBeanName())) {
                    try {
                        Class beanClass = Class.forName(beanDefinition.getBeanClassName());
                        if (type.isAssignableFrom(beanClass)) {
                            result.put(beanDefinitionName, (T) getBean(beanDefinitionName));
                        }
                    } catch (ClassNotFoundException e) {
                        e.printStackTrace();
                    }
                }

            }
        }
        return result;
    }

    public Map<String, BeanDefinition> getBeanDefinitionsToRegister() {
        return beanDefinitionsToRegister;
    }

    public Class getEndpointFactoryClass() {
        return endpointFactoryClass;
    }

    public DOMParser getMunitDomParser() {
        return munitDomParser;
    }

    public String getMunitFactoryPostProcessorId() {
        return munitFactoryPostProcessorId;
    }
}
