/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.api.spring.config.parsers;

import org.apache.commons.lang.StringUtils;
import org.mule.api.config.MuleProperties;
import org.mule.config.spring.parsers.generic.OrphanDefinitionParser;
import org.mule.config.spring.util.ProcessingStrategyUtils;
import org.springframework.beans.factory.config.ConfigurableBeanFactory;
import org.springframework.beans.factory.config.RuntimeBeanReference;
import org.springframework.beans.factory.support.BeanDefinitionBuilder;
import org.springframework.beans.factory.xml.ParserContext;
import org.w3c.dom.Element;

import java.util.HashMap;
import java.util.Map;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;
import static org.apache.commons.lang.StringUtils.isNotBlank;

public class MunitOrphanDefinitionParser extends OrphanDefinitionParser implements MunitDefinitionParser {

    private Class messageProcessorClass;

    private String scope;
    private Map<String, String> attributePropertyMap;

    private MunitOrphanDefinitionParser(Class messageProcessorClass, String scope, Map<String, String> attributePropertyMap) {
        super(messageProcessorClass, true);
        addIgnored("abstract");
        addIgnored("name");
        addIgnored("processingStrategy");

        this.messageProcessorClass = messageProcessorClass;
        this.scope = scope;
        this.attributePropertyMap = attributePropertyMap;
    }

    public static class MunitOrphanDefinitionParserBuilder {
        private Class messageProcessorClass;

        protected String scope;
        protected Map<String, String> attributePropertyMap;

        public MunitOrphanDefinitionParserBuilder() {
            this.attributePropertyMap = new HashMap<>();
        }

        public MunitOrphanDefinitionParserBuilder withMessageProcessorClass(Class messageProcessorClass) {
            checkNotNull(messageProcessorClass, "The message processor class must not be null.");
            this.messageProcessorClass = messageProcessorClass;
            return this;
        }

        public MunitOrphanDefinitionParserBuilder withScope(String scope) {
            checkArgument(isNotBlank(scope), "The scope must not be null nor empty.");
            if (!scope.equals(ConfigurableBeanFactory.SCOPE_PROTOTYPE) && !scope.equals(ConfigurableBeanFactory.SCOPE_SINGLETON)) {
                throw new IllegalArgumentException("The provided scope is not valid.");
            }

            this.scope = scope;
            return this;
        }

        public MunitOrphanDefinitionParserBuilder withAttribute(String attributeName, String propertyName) {
            checkArgument(isNotBlank(attributeName), "The attribute name must not be null nor empty.");
            checkArgument(isNotBlank(propertyName), "The property name must not be null nor empty.");
            this.attributePropertyMap.put(attributeName, propertyName);
            return this;
        }

        public MunitOrphanDefinitionParser build() {
            checkNotNull(messageProcessorClass, "Build Fail. The message processor class can not be null.");
            checkArgument(isNotBlank(scope), "Build Fail.The scope can not be null nor empty.");

            MunitOrphanDefinitionParser parser = new MunitOrphanDefinitionParser(messageProcessorClass, scope, attributePropertyMap);

            return parser;
        }
    }

    @Override
    protected void doParse(Element element, ParserContext parserContext, BeanDefinitionBuilder builder) {
        builder.addConstructorArgValue(element.getAttribute(ATTRIBUTE_NAME));
        builder.addConstructorArgReference(MuleProperties.OBJECT_MULE_CONTEXT);

        parseMessageProcessorAttributes(element, builder);

        ProcessingStrategyUtils.configureProcessingStrategy(element, builder, ProcessingStrategyUtils.QUEUED_ASYNC_PROCESSING_STRATEGY);

        super.doParse(element, parserContext, builder);
    }

    protected void parseMessageProcessorAttributes(Element element, BeanDefinitionBuilder builder) {
        parseElementAttributes(this.attributePropertyMap, element, builder);
    }


    // TODO: this is the same as the generic we need to externilize such methods or make one static
    protected void parseElementAttributes(Map<String, String> attributePropetyMap, Element element, BeanDefinitionBuilder builder) {
        for (String attributeName : attributePropetyMap.keySet()) {
            String propertyName = attributePropetyMap.get(attributeName);

            if (hasAttribute(element, attributeName)) {
                String attributeValue = element.getAttribute(attributeName);
                builder.addPropertyValue(propertyName, buildAttributeValue(attributeName, attributeValue));
            }
        }
    }

    // TODO: this is the same as the generic we need to externilize such methods or make one static
    protected boolean hasAttribute(Element element, String attributeName) {
        String value = element.getAttribute(attributeName);
        if ((value != null) && (!StringUtils.isBlank(value))) {
            return true;
        }
        return false;
    }

    // TODO: this is the same as the generic we need to externilize such methods or make one static
    protected Object buildAttributeValue(String attributeName, String attributeValue) {
        if (attributeName.endsWith("-ref")) {
            if (attributeValue.startsWith("#[")) {
                return attributeValue;
            } else {
                return new RuntimeBeanReference(attributeValue);
            }
        } else {
            return attributeValue;
        }
    }


    public Class getMessageProcessorClass() {
        return messageProcessorClass;
    }

    public String getScope() {
        return scope;
    }

    public Map<String, String> getAttributePropertyMap() {
        return attributePropertyMap;
    }
}
