/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.runner.spring.config.reader;

import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.mule.config.spring.MuleHierarchicalBeanDefinitionParserDelegate;
import org.mule.config.spring.util.SpringXMLUtils;
import org.springframework.beans.factory.config.BeanDefinition;
import org.springframework.beans.factory.config.BeanDefinitionHolder;
import org.springframework.beans.factory.xml.DefaultBeanDefinitionDocumentReader;
import org.springframework.beans.factory.xml.NamespaceHandler;
import org.springframework.beans.factory.xml.NamespaceHandlerResolver;
import org.springframework.beans.factory.xml.ParserContext;
import org.springframework.beans.factory.xml.XmlReaderContext;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;


/**
 * <p>
 * The {@link MunitBeanDefinitionParserDelegate} that is an extension of the {@link org.mule.config.spring.MuleHierarchicalBeanDefinitionParserDelegate}
 * it overrides only the parseCustomElement to wrapp then {@link org.springframework.beans.factory.xml.NamespaceHandler}
 * </p>
 *
 * @author Mulesoft Inc.
 * @since 3.3.2
 */
public class MunitBeanDefinitionParserDelegate extends MuleHierarchicalBeanDefinitionParserDelegate {

    public static final String BEANS = "beans"; // cannot find this in Spring api!
    private static final String DOCUMENT_BUILDER_FACTORY = "org.apache.xerces.jaxp.DocumentBuilderFactoryImpl";

    private DefaultBeanDefinitionDocumentReader spring;

    public MunitBeanDefinitionParserDelegate(XmlReaderContext readerContext, DefaultBeanDefinitionDocumentReader spring) {
        super(readerContext, spring);
        this.spring = spring;
    }

    protected  NamespaceHandler getMunitHandlerWrapper(NamespaceHandler handler){
        return new MunitHandlerWrapper(handler);
    }

    @Override
    public BeanDefinition parseCustomElement(Element element, BeanDefinition parent) {
        if (logger.isDebugEnabled()) {
            logger.debug("parsing: " + SpringXMLUtils.elementToString(element));
        }
        if (SpringXMLUtils.isBeansNamespace(element)) {
            return handleSpringElements(element, parent);
        } else {
            String namespaceUri = element.getNamespaceURI();
            NamespaceHandlerResolver namespaceHandlerResolver = getReaderContext().getNamespaceHandlerResolver();
            NamespaceHandler handler = namespaceHandlerResolver.resolve(namespaceUri);
            if (handler == null) {
                getReaderContext().error("Unable to locate NamespaceHandler for namespace [" + namespaceUri + "]", element);
                return null;
            }

            boolean noRecurse = false;
            boolean forceRecurse = false;
            BeanDefinition finalChild;

            do {
                ParserContext parserContext = new ParserContext(getReaderContext(), this, parent);
                finalChild = getMunitHandlerWrapper(handler).parse(element, parserContext);
                registerBean(element, finalChild);
                noRecurse = noRecurse || testFlag(finalChild, MULE_NO_RECURSE);
                forceRecurse = forceRecurse || testFlag(finalChild, MULE_FORCE_RECURSE);
            }
            while (null != finalChild && testFlag(finalChild, MULE_REPEAT_PARSE));


            boolean isRecurse;
            if (noRecurse) {
                // no recursion takes precedence, as recursion is set by default
                isRecurse = false;
            } else {
                if (forceRecurse) {
                    isRecurse = true;
                } else {
                    // default behaviour if no control specified
                    isRecurse = SpringXMLUtils.isMuleNamespace(element);
                }
            }

            if (isRecurse) {
                NodeList list = element.getChildNodes();
                for (int i = 0; i < list.getLength(); i++) {
                    if (list.item(i) instanceof Element) {
                        parseCustomElement((Element) list.item(i), finalChild);
                    }
                }
            }

            // If a parsers requests post-processing we call again after children called

            if (testFlag(finalChild, MULE_POST_CHILDREN)) {
                ParserContext parserContext = new ParserContext(getReaderContext(), this, parent);
                finalChild = handler.parse(element, parserContext);
            }

            return finalChild;
        }
    }

    /**
     * Since mule runtime 3.9.x, we have to force the xerces implementation of the {@link DocumentBuilderFactory} to avoid a jar
     * hell between other xml libraries.
     */
    @Override
    protected BeanDefinition handleSpringElements(Element element, BeanDefinition parent) {
        if (SpringXMLUtils.isLocalName(element, BEANS)) {
            try {
                DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance(DOCUMENT_BUILDER_FACTORY, getClass().getClassLoader());
                Document doc = factory.newDocumentBuilder().newDocument();
                doc.appendChild(doc.importNode(element, true));
                this.spring.registerBeanDefinitions(doc, this.getReaderContext());
                return parent;
            } catch (ParserConfigurationException var5) {
                throw new RuntimeException(var5);
            }
        } else if (SpringXMLUtils.isLocalName(element, PROPERTY_ELEMENT)) {
            this.parsePropertyElement(element, parent);
            return parent;
        } else if (SpringXMLUtils.isLocalName(element, BEAN_ELEMENT)) {
            BeanDefinitionHolder holder = this.parseBeanDefinitionElement(element, parent);
            this.registerBeanDefinitionHolder(holder);
            return holder.getBeanDefinition();
        } else {
            throw new IllegalStateException("Unexpected Spring element: " + SpringXMLUtils.elementToString(element));
        }
    }

}
