/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.common.processor.interceptor;

import net.sf.cglib.proxy.MethodProxy;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.mule.DefaultMuleEvent;
import org.mule.RequestContext;
import org.mule.api.MuleContext;
import org.mule.api.MuleEvent;
import org.mule.api.processor.MessageProcessor;
import org.mule.modules.interceptor.processors.AbstractMessageProcessorInterceptor;
import org.mule.modules.interceptor.processors.MessageProcessorBehavior;
import org.mule.munit.common.MunitUtils;
import org.mule.munit.common.mocking.SamePayload;
import org.mule.munit.common.processor.MockedMessageProcessorManager;
import org.mule.munit.common.processor.MunitMessageProcessorCall;
import org.mule.munit.common.processor.SpyAssertion;
import org.mule.munit.common.processors.InterceptingMessageProcessorHandler;

/**
 * <p>
 * It intercepts the {@link MessageProcessor#process(org.mule.api.MuleEvent)}  calls
 * </p>
 *
 * @author Mulesoft Inc.
 * @since 3.3.2
 */
public class MunitMessageProcessorInterceptor extends AbstractMessageProcessorInterceptor {
    private transient Log logger = LogFactory.getLog(this.getClass());

    private String fileName;
    private String lineNumber;

    public Object process(Object obj, Object[] args, MethodProxy proxy) throws Throwable {
        MuleEvent originalEvent = (MuleEvent) args[0];

        EventCopyManager eventCopyManager = new EventCopyManager(originalEvent);
        MuleEvent copyEvent = eventCopyManager.getCopyEvent();

        MockedMessageProcessorManager manager = getMockedMessageProcessorManager(copyEvent.getMuleContext());

        MunitMessageProcessorCall messageProcessorCall = buildCall(obj, copyEvent);
        logger.debug("Executing MP: " + messageProcessorCall.getMessageProcessorId().getFullName() + "-[" + fileName + "|" + lineNumber + "]");

        logger.debug("About to run Spy Before...");
        runSpyAssertion(manager.getBetterMatchingBeforeSpyAssertion(messageProcessorCall), copyEvent);

        registerCall(manager, messageProcessorCall);
        MessageProcessorBehavior behavior = manager.getBetterMatchingBehavior(messageProcessorCall);
        if (behavior != null) {
            logger.debug("Mock behavior found for message processor");

            if (behavior.getExceptionToThrow() != null) {
                logger.debug("Mock behavior will throw exception");
                runSpyAssertion(manager.getBetterMatchingAfterSpyAssertion(messageProcessorCall), copyEvent);
                throw behavior.getExceptionToThrow();
            }

            if (behavior.getMuleMessageTransformer() != null) {
                copyEvent.setMessage(behavior.getMuleMessageTransformer().transform(copyEvent.getMessage()));
                Object payload = copyEvent.getMessage().getPayload();
                if (payload instanceof SamePayload) {
                    SamePayload samePayload = (SamePayload) payload;
                    copyEvent.getMessage().setPayload(samePayload.process(originalEvent));
                }
            }

            logger.debug("About to run Spy After...");
            runSpyAssertion(manager.getBetterMatchingAfterSpyAssertion(messageProcessorCall), copyEvent);

            return handleInterceptingMessageProcessors(obj, copyEvent);
        }

        logger.debug("No mock behavior found for message processor, invoking original message processor");
        eventCopyManager.rollBackRequestContext();
        Object o = invokeSuper(obj, args, proxy);

        logger.debug("About to run Spy After...");
        runSpyAssertion(manager.getBetterMatchingAfterSpyAssertion(messageProcessorCall), (MuleEvent) o);
        return o;
    }

    protected Object handleInterceptingMessageProcessors(Object obj, MuleEvent event) throws Throwable {
        return new InterceptingMessageProcessorHandler(obj).invokeProcessNext(event);
    }


    protected Object invokeSuper(Object obj, Object[] args, MethodProxy proxy) throws Throwable {
        return proxy.invokeSuper(obj, args);
    }


    private void registerCall(MockedMessageProcessorManager manager, MunitMessageProcessorCall messageProcessorCall) {
        manager.addCall(messageProcessorCall);
    }

    private void runSpyAssertion(SpyAssertion spyAssertion, MuleEvent event) {
        if (spyAssertion == null) {
            logger.debug("No Spy was found to be run.");
            return;
        }

        logger.debug("Running Spy");
        MunitUtils.verifyAssertions(event, spyAssertion.getMessageProcessors());
    }

    private MunitMessageProcessorCall buildCall(Object originalMp, MuleEvent event) {
        MunitMessageProcessorCall call = new MunitMessageProcessorCall(id);

//        call.setAttributes(getAttributes(event));
        call.setAttributes(MessageProcessorAttributesEvaluator.getEvaluatedAttributes(originalMp, attributes, event));

        call.setFlowConstruct(event.getFlowConstruct());
        call.setFileName(fileName);
        call.setLineNumber(lineNumber);
        return call;
    }


    protected MockedMessageProcessorManager getMockedMessageProcessorManager(MuleContext muleContext) {
        return ((MockedMessageProcessorManager) muleContext.getRegistry().lookupObject(MockedMessageProcessorManager.ID));
    }


    public String getFileName() {
        return fileName;
    }

    public void setFileName(String fileName) {
        this.fileName = fileName;
    }

    public void setLineNumber(String lineNumber) {
        this.lineNumber = lineNumber;
    }


    public static class EventCopyManager {
        private MuleEvent originalEvent = null;
        private MuleEvent copyEvent = null;

        public EventCopyManager(MuleEvent originalEvent) {
            this.originalEvent = originalEvent;
        }

        public MuleEvent getCopyEvent() {
            if (null == copyEvent) {
                this.copyEvent = DefaultMuleEvent.copy(this.originalEvent);
                RequestContext.setEvent(this.copyEvent);
            }
            return this.copyEvent;
        }

        public void rollBackRequestContext() {
            RequestContext.setEvent(this.originalEvent);
        }

    }
}
