/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.common.processor.interceptor;

import static java.util.Arrays.asList;
import static java.util.regex.Pattern.matches;

import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.xml.stream.XMLStreamReader;

import org.apache.commons.lang3.StringUtils;
import org.mule.api.MuleEvent;
import org.mule.util.AttributeEvaluator;

public class MessageProcessorAttributesEvaluator {

    private static final List<String> PAYLOAD_TRANSFORMATION_EXPRESSIONS = asList( ".*#\\[.*message.payloadAs\\(.*\\].*", ".*#\\[payload:.*\\].*", ".*#\\[dw\\(.*payload.*\\)\\].*");
    private static final List<String> PAYLOAD_ITERATION_EXPRESSIONS = asList(".*#\\[.*payload.next\\(\\).*\\].*");

    /**
     * The goal of this method is, provided a @originalMp, a list of @attributes and an @event evaluate the values of those attributes for the current event.
     * But before that filter those attributes of the mp that should not be evaluated.
     *
     * @param originalMp the mp that owns the attributes
     * @param attributes the message processor's attributes with its values (not evaluated)
     * @param event      the current event as context to evaluate the attributes
     * @return a map containing the message processor's attributes and its evaluated values
     */
    protected static Map<String, Object> getEvaluatedAttributes(Object originalMp, Map<String, String> attributes, MuleEvent event) {

        Map<String, String> filteredAttributes = MessageProcessorAttributeFilter.filterAttributes(originalMp, attributes);

        Map<String, Object> processed = new HashMap<String, Object>();
        for (Map.Entry<String, String> attrs : filteredAttributes.entrySet()) {
            try {
                if(shouldEvaluate(event, attrs.getValue())){
                    Object evaluate = evaluate(attrs.getValue(), event);
                    processed.put(attrs.getKey(), evaluate);
                } else {
                    processed.put(attrs.getKey(), attrs.getValue());
                }
            } catch (Throwable t) {
                processed.put(attrs.getKey(), attrs.getValue());
            }
        }
        return processed;
    }

    private static boolean shouldEvaluate(MuleEvent event, String value) {
        Object payload = event.getMessage().getPayload();
        return (!isPayloadStreamTransformation(payload, value) && !isIteratorPayloadIteration(payload, value));
    }

    private static boolean isIteratorPayloadIteration(Object payload, String value) {
        return (payload instanceof Iterator) && isPayloadIteration(value);
    }

    private static boolean isPayloadStreamTransformation(Object payload, String value) {
        return (payload instanceof XMLStreamReader) && isPayloadTransformation(value);
    }

    /**
     * This method is to avoid evaluating expressions that iterate the payload, because when reading it again during the flow execution
     * the payload would be already iterated.
     */
    private static boolean isPayloadIteration(String expression) {
        for (String payloadTransformationRegex : PAYLOAD_ITERATION_EXPRESSIONS) {
            if (matches(payloadTransformationRegex, expression)) {
                return true;
            }
        }
        return false;
    }

    /**
     * This method is to avoid evaluating expressions that read the payload, because when reading it again during the flow
     * execution the payload would be already read.
     */
    private static boolean isPayloadTransformation(String expression) {
        for(String payloadTransformationRegex : PAYLOAD_TRANSFORMATION_EXPRESSIONS){
            if(matches(payloadTransformationRegex, expression)) {
                return true;
            }
        }
        return false;
    }

    private static Object evaluate(String elementValue, MuleEvent event) {
        Object compareTo = elementValue;
        AttributeEvaluator attributeEvaluator = getAttributeEvaluator(elementValue, event);
        if (attributeEvaluator.isExpression() || attributeEvaluator.isParseExpression()) {
            compareTo = attributeEvaluator.resolveValue(event);
        } else if (!StringUtils.isEmpty(elementValue)) {
            Object o = event.getMuleContext().getRegistry().lookupObject(elementValue);
            if (o != null) {
                compareTo = o;
            }
        }
        return compareTo;
    }

    private static AttributeEvaluator getAttributeEvaluator(String attributeValue, MuleEvent event) {
        AttributeEvaluator attributeEvaluator = new AttributeEvaluator(attributeValue);
        attributeEvaluator.initialize(event.getMuleContext().getExpressionManager());
        return attributeEvaluator;
    }
}
