/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.common.processor.interceptor;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class MessageProcessorAttributeFilter {
    private static transient Log logger = LogFactory.getLog(MessageProcessorAttributeFilter.class);

    private static Map<String, List<String>> classAttributeMap;

    /**
     * The method will filter the attributes in the @attributes map based on the class of the @messageProcessor.
     * To decide what attributes should be filtered the class has an internal map where it's registered which class to filter
     * out  and for that class which attributes.
     *
     * @param messageProcessor the message processor instance
     * @param attributes       list of message processor attributes and its values
     * @return a Map containing the message processor's attributes filtered out
     */
    public static Map<String, String> filterAttributes(Object messageProcessor, Map<String, String> attributes) {
        initIfNecessary();

        Map<String, String> filteredAttribtues = new HashMap<String, String>();
        filteredAttribtues.putAll(attributes);

        for (String className : classAttributeMap.keySet()) {
            try {
                Class clazz = Class.forName(className);
                if (clazz.isAssignableFrom(messageProcessor.getClass())) {
                    for (String a : classAttributeMap.get(className)) {
                        if (filteredAttribtues.containsKey(a)) {
                            filteredAttribtues.remove(a);
                        }
                    }
                }
            } catch (ClassNotFoundException e) {
                logger.warn("Filter unwanted attributes from mp failed. No class could be found for name: " + className);
                continue;
            }
        }

        return filteredAttribtues;
    }

    private static void initIfNecessary() {
        if (classAttributeMap == null) {
            classAttributeMap = new HashMap<String, List<String>>();
            addForEachAttributes();
        }
    }

    private static void addForEachAttributes() {
        String className = "org.mule.routing.Foreach";

        List<String> attributesToFilter = new ArrayList<String>();
        attributesToFilter.add("name");

        classAttributeMap.put(className, attributesToFilter);
    }

}
