/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.common.processor.interceptor;

import net.sf.cglib.proxy.MethodProxy;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.mule.DefaultMuleEvent;
import org.mule.RequestContext;
import org.mule.api.MuleContext;
import org.mule.api.MuleEvent;
import org.mule.api.MuleMessage;
import org.mule.api.processor.MessageProcessor;
import org.mule.modules.interceptor.processors.AbstractMessageProcessorInterceptor;
import org.mule.modules.interceptor.processors.MessageProcessorBehavior;
import org.mule.munit.common.MunitUtils;
import org.mule.munit.common.adapters.MuleMessageDataTypeSetterAdapter;
import org.mule.munit.common.mocking.SamePayload;
import org.mule.munit.common.processor.MockedMessageProcessorManager;
import org.mule.munit.common.processor.MunitMessageProcessorCall;
import org.mule.munit.common.processor.SpyAssertion;
import org.mule.munit.common.processors.InterceptingMessageProcessorHandler;
import org.mule.munit.common.util.ReusableByteArrayInputStream;

import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.InvocationTargetException;

/**
 * <p>
 * It intercepts the {@link MessageProcessor#process(org.mule.api.MuleEvent)} calls
 * </p>
 *
 * @author Mulesoft Inc.
 * @since 3.3.2
 */
public class MunitMessageProcessorInterceptor extends AbstractMessageProcessorInterceptor {

  private static transient Log logger = LogFactory.getLog(MunitMessageProcessorInterceptor.class);

  private String fileName;
  private String lineNumber;
  private ThreadLocal<EventCopyManager> eventCopyManager = new ThreadLocal<EventCopyManager>();

  public Object process(Object obj, Object[] args, MethodProxy proxy) throws Throwable {
    MuleEvent originalEvent = (MuleEvent) args[0];
    MuleEvent copyEvent = getCopyEvent(originalEvent);
    Object payloadCopy = originalEvent.getMessage().getPayload();
    MockedMessageProcessorManager manager = getMockedMessageProcessorManager(copyEvent.getMuleContext());
    MunitMessageProcessorCall messageProcessorCall = buildCall(obj, copyEvent);

    logger.debug(
                 "Executing MP: " + messageProcessorCall.getMessageProcessorId().getFullName() + "-[" + fileName + "|"
                     + lineNumber
                     + "]");

    logger.debug("About to run Spy Before...");
    runSpyAssertion(manager.getBetterMatchingBeforeSpyAssertion(messageProcessorCall), originalEvent);
    registerCall(manager, messageProcessorCall);

    MessageProcessorBehavior behavior = manager.getBetterMatchingBehavior(messageProcessorCall);

    if (behavior != null) {
      logger.debug("Mock behavior found for message processor");
      originalEvent = invokeBehavior(behavior, manager, messageProcessorCall, originalEvent, payloadCopy);
      return handleInterceptingMessageProcessors(obj, originalEvent);
    } else {
      logger.debug("No mock behavior found for message processor, invoking original message processor");
      eventCopyManager.get().rollBackRequestContext();

      Object o = invokeSuper(obj, args, proxy);

      logger.debug("About to run Spy After...");
      runSpyAssertion(manager.getBetterMatchingAfterSpyAssertion(messageProcessorCall), originalEvent);
      return o;
    }
  }

  private void buildEventCopyManager(MuleEvent originalEvent) {
    this.eventCopyManager.set(new EventCopyManager(originalEvent));
  }

  private MuleEvent invokeBehavior(MessageProcessorBehavior behavior, MockedMessageProcessorManager manager,
                                   MunitMessageProcessorCall messageProcessorCall, MuleEvent originalEvent, Object payloadCopy)
      throws Throwable {
    boolean shouldReturnSamePayload = false;
    if (behavior.getExceptionToThrow() != null) {
      logger.debug("Mock behavior will throw exception");
      runSpyAssertion(manager.getBetterMatchingAfterSpyAssertion(messageProcessorCall), originalEvent);
      throw behavior.getExceptionToThrow();
    }
    if (behavior.getMuleMessageTransformer() != null) {
      behavior.getMuleMessageTransformer().transform(originalEvent.getMessage());
      shouldReturnSamePayload = originalEvent.getMessage().getPayload() instanceof SamePayload;
    }
    runSpyAssertion(manager.getBetterMatchingAfterSpyAssertion(messageProcessorCall), originalEvent);
    if (shouldReturnSamePayload) {
      originalEvent.getMessage().setPayload(payloadCopy);
    }
    return originalEvent;
  }


  protected Object handleInterceptingMessageProcessors(Object obj, MuleEvent event) throws Throwable {
    try {
      return new InterceptingMessageProcessorHandler(obj).invokeProcessNext(event);
    } catch (InvocationTargetException e) {
      if (e.getTargetException() instanceof UnsupportedOperationException) {
        logger.debug("An UnsupportedOperationException was thrown when invoking the 'processNext' method",
                     e.getTargetException());
        return event;
      } else {
        throw e;
      }
    }
  }


  protected Object invokeSuper(Object obj, Object[] args, MethodProxy proxy) throws Throwable {
    return proxy.invokeSuper(obj, args);
  }


  private void registerCall(MockedMessageProcessorManager manager, MunitMessageProcessorCall messageProcessorCall) {
    manager.addCall(messageProcessorCall);
  }

  private void runSpyAssertion(SpyAssertion spyAssertion, MuleEvent event) {
    if (spyAssertion == null) {
      logger.debug("No Spy was found to be run.");
      return;
    }

    logger.debug("Running Spy");
    MunitUtils.verifyAssertions(event, spyAssertion.getMessageProcessors());
  }

  private MunitMessageProcessorCall buildCall(Object originalMp, MuleEvent event) {
    MunitMessageProcessorCall call = new MunitMessageProcessorCall(id);

    call.setAttributes(MessageProcessorAttributesEvaluator.getEvaluatedAttributes(originalMp, attributes, event));

    call.setFlowConstruct(event.getFlowConstruct());
    call.setFileName(fileName);
    call.setLineNumber(lineNumber);
    return call;
  }


  protected MockedMessageProcessorManager getMockedMessageProcessorManager(MuleContext muleEvent) {
    return ((MockedMessageProcessorManager) muleEvent.getRegistry()
        .lookupObject(MockedMessageProcessorManager.ID));
  }


  public String getFileName() {
    return fileName;
  }

  public void setFileName(String fileName) {
    this.fileName = fileName;
  }

  public void setLineNumber(String lineNumber) {
    this.lineNumber = lineNumber;
  }

  public MuleEvent getCopyEvent(MuleEvent originalEvent) {
    buildEventCopyManager(originalEvent);
    return eventCopyManager.get().getCopyEvent();
  }

  public static class EventCopyManager {

    private MuleEvent originalEvent = null;
    private MuleEvent copyEvent = null;

    public EventCopyManager(MuleEvent originalEvent) {
      this.originalEvent = originalEvent;
    }

    public MuleEvent getCopyEvent() {
      if (null == copyEvent) {
        this.originalEvent = replacePayloadIfInputSteam();
        this.copyEvent = DefaultMuleEvent.copy(this.originalEvent);
        MunitUtils.copyMessage(this.originalEvent.getMessage(), this.copyEvent.getMessage());
        RequestContext.setEvent(this.copyEvent);
      }
      return this.copyEvent;
    }

    /**
     * We need to replace the original InputStream with a ReusableInputStream in order to be able to evaluate the payload and let
     * the stream be correctly read afterwards. If the payload is not an InputStream, it returns the same event
     */
    private MuleEvent replacePayloadIfInputSteam() {
      MuleMessage message = this.originalEvent.getMessage();
      if (message.getPayload() instanceof InputStream) {
        InputStream originalStream = (InputStream) message.getPayload();
        try {
          InputStream reusableStream = ReusableByteArrayInputStream.of(originalStream);
          MuleMessageDataTypeSetterAdapter muleMessageDataTypeSetterAdapter =
              new MuleMessageDataTypeSetterAdapter(message);
          muleMessageDataTypeSetterAdapter.setPayload(reusableStream, message.getDataType());
        } catch (IOException e) {
          logger.warn("Unable to replace InputStream with ReusableByteArrayInputStream. Event will remain the same", e);
        }
      }
      return this.originalEvent;
    }

    public void rollBackRequestContext() {
      RequestContext.setEvent(this.originalEvent);
    }

  }
}
