/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.common.processor;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.mule.modules.interceptor.processors.MessageProcessorCall;
import org.mule.modules.interceptor.processors.MessageProcessorId;
import org.mule.modules.interceptor.processors.MessageProcessorManager;

import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CopyOnWriteArrayList;

/**
 * <p>
 * The Class that manages the mocking process. Gets the behaviors, stores the message processor calls and stores
 * the spy process
 * </p>
 *
 * @author Mulesoft Inc.
 * @since 3.3.2
 */
public class MockedMessageProcessorManager extends MessageProcessorManager {
    public static String ID = "_muleMockMpManager";

    private transient Log logger = LogFactory.getLog(this.getClass());

    /**
     * <p>
     * These are the real calls of the message processors.
     * </p>
     */
    protected List<MunitMessageProcessorCall> calls = new CopyOnWriteArrayList<MunitMessageProcessorCall>();


    /**
     * <p>
     * The spy assertions that are ran before a message processor call
     * </p>
     */
    protected List<SpyAssertion> beforeCallSpyAssertions = new CopyOnWriteArrayList<SpyAssertion>();

    /**
     * <p>
     * The spy assertions that are ran after a message processor call
     * </p>
     */
    protected List<SpyAssertion> afterCallSpyAssertions = new CopyOnWriteArrayList<SpyAssertion>();

    /**
     * <p>
     * Reset all the status
     * </p>
     */
    public void reset() {
        behaviors.clear();
        calls.clear();
        beforeCallSpyAssertions.clear();
        afterCallSpyAssertions.clear();
    }

    /**
     * <p>
     * Retrieve all the execute calls for a message processor that satisfies the attribute matchers
     * </p>
     *
     * @param mpId               The Message processor Id
     * @param attributesMatchers The attributes that the message processor must match
     * @return The List of message processor calls
     */
    public List<MessageProcessorCall> findCallsFor(MessageProcessorId mpId, Map<String, Object> attributesMatchers) {
        logger.debug("Finding calls for message processor: " + mpId.getFullName());

        List<MessageProcessorCall> expected = new ArrayList<MessageProcessorCall>();
        MessageProcessorCall matchingCall = new MessageProcessorCall(mpId);
        matchingCall.setAttributes(attributesMatchers);
        for (MessageProcessorCall call : calls) {
            if (matchingCall.matchingWeight(call) >= 0) {
                expected.add(call);
            }
        }
        return expected;
    }

    /**
     * <p>
     * Gets the best matching Before Spy assertion.
     * </p>
     *
     * @param messageProcessorCall The comparing call
     * @return The best matching Before spy assertion
     */
    public SpyAssertion getBetterMatchingBeforeSpyAssertion(MessageProcessorCall messageProcessorCall) {
        logger.debug("Finding Spy Before for: " + messageProcessorCall.getMessageProcessorId().getFullName());
        return getBetterMatchingAction(messageProcessorCall, beforeCallSpyAssertions);
    }


    /**
     * <p>
     * Gets the best matching After Spy assertion.
     * </p>
     *
     * @param messageProcessorCall The comparing call
     * @return The best matching After spy assertion
     */
    public SpyAssertion getBetterMatchingAfterSpyAssertion(MessageProcessorCall messageProcessorCall) {
        logger.debug("Finding Spy After for: " + messageProcessorCall.getMessageProcessorId().getFullName());
        return getBetterMatchingAction(messageProcessorCall, afterCallSpyAssertions);
    }

    public synchronized void addCall(MunitMessageProcessorCall call) {
        calls.add(call);
    }

    public synchronized void addBeforeCallSpyAssertion(SpyAssertion spyAssertion) {
        beforeCallSpyAssertions.add(spyAssertion);
    }

    public synchronized void addAfterCallSpyAssertion(SpyAssertion spyAssertion) {
        afterCallSpyAssertions.add(spyAssertion);
    }

    public List<MunitMessageProcessorCall> getCalls() {
        return new LinkedList<MunitMessageProcessorCall>(calls);
    }
}
