/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.common.processor.interceptor;

import org.apache.commons.lang.StringUtils;
import org.mule.api.MuleEvent;
import org.mule.util.AttributeEvaluator;

import java.util.HashMap;
import java.util.Map;

public class MessageProcessorAttributesEvaluator {

    /**
     * The goal of this method is, provided a @originalMp, a list of @attributes and an @event evaluate the values of those attributes for the current event.
     * But before that filter those attributes of the mp that should not be evaluated.
     *
     * @param originalMp the mp that owns the attributes
     * @param attributes the message processor's attributes with its values (not evaluated)
     * @param event      the current event as context to evaluate the attributes
     * @return a map containing the message processor's attributes and its evaluated values
     */
    protected static Map<String, Object> getEvaluatedAttributes(Object originalMp, Map<String, String> attributes, MuleEvent event) {

        Map<String, String> filteredAttributes = MessageProcessorAttributeFilter.filterAttributes(originalMp, attributes);

        Map<String, Object> processed = new HashMap<String, Object>();
        for (Map.Entry<String, String> attrs : filteredAttributes.entrySet()) {
            try {
                Object evaluate = evaluate(attrs.getValue(), event);
                processed.put(attrs.getKey(), evaluate);
            } catch (Throwable t) {
                processed.put(attrs.getKey(), attrs.getValue());
            }
        }
        return processed;
    }

    private static Object evaluate(String elementValue, MuleEvent event) {
        Object compareTo = elementValue;
        AttributeEvaluator attributeEvaluator = getAttributeEvaluator(elementValue, event);
        if (attributeEvaluator.isExpression() || attributeEvaluator.isParseExpression()) {
            compareTo = attributeEvaluator.resolveValue(event);
        } else if (!StringUtils.isEmpty(elementValue)) {
            Object o = event.getMuleContext().getRegistry().lookupObject(elementValue);
            if (o != null) {
                compareTo = o;
            }
        }
        return compareTo;
    }

    private static AttributeEvaluator getAttributeEvaluator(String attributeValue, MuleEvent event) {
        AttributeEvaluator attributeEvaluator = new AttributeEvaluator(attributeValue);
        attributeEvaluator.initialize(event.getMuleContext().getExpressionManager());
        return attributeEvaluator;
    }
} 