/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.common.processor.interceptor;

import net.sf.cglib.proxy.*;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.mule.api.MuleEvent;
import org.mule.api.processor.MessageProcessor;
import org.mule.modules.interceptor.processors.MessageProcessorId;
import org.mule.munit.common.processor.MunitNamingPolicy;
import org.mule.munit.runner.interceptor.AbstractMessageProcessorInterceptorFactory;
import org.springframework.beans.factory.FactoryBean;

import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.Map;

//@TODO review: FactoryBean - AbstractBeanDefinition

/**
 * <p>
 * This is the Message processor interceptor factory.
 * </p>
 *
 * @author Mulesoft Inc.
 * @since 3.3.2
 */
public class MunitMessageProcessorInterceptorFactory extends AbstractMessageProcessorInterceptorFactory {

    protected transient Log logger = LogFactory.getLog(getClass());

    /**
     * <p>
     * For operations that are not {@link org.mule.api.processor.MessageProcessor#process(org.mule.api.MuleEvent)} just do
     * nothing
     * </p>
     */
    private static Callback NULL_METHOD_INTERCEPTOR = new NoOp() {
    };

    private static CallbackFilter FACTORY_BEAN_FILTER = new CallbackFilter() {

        @Override
        public int accept(Method method) {
            if ("getObject".equals(method.getName())) {
                return 0;
            }
            return 1;
        }
    };

    private static CallbackFilter MESSAGE_PROCESSOR_FILTER = new CallbackFilter() {

        @Override
        public int accept(Method method) {
            Class<?> declaringClass = method.getDeclaringClass();
            if (MessageProcessor.class.isAssignableFrom(declaringClass) && method.getName().equals("process") && method.getParameterTypes().length == 1 && MuleEvent.class.isAssignableFrom(method.getParameterTypes()[0])) {
                return 0;
            }
            return 1;
        }
    };

    private Object createRealMpInstance(Class realMpClass, MessageProcessorId id, Object[] constructorArguments) throws NoSuchMethodException, IllegalAccessException, InvocationTargetException, InstantiationException {

        if (constructorArguments != null && constructorArguments.length != 0) {
            Class[] classes = findConstructorArgumentTypes(realMpClass, constructorArguments);
            if (classes != null) {
                Constructor constructor = realMpClass.getConstructor(classes);
                return constructor.newInstance(constructorArguments);
            } else {
                logger.warn("The message processor " + id.getFullName() + " has no matching constructor for the offered parameters creating it with default constructor");
                return realMpClass.newInstance();
            }

        } else {
            return realMpClass.newInstance();
        }
    }

    private Class getPrimitiveWrapperClass(Class clazz) {

        String primitiveName = clazz.toString();

        if ("boolean".equals(primitiveName)) {
            return Boolean.class;
        }

        if ("byte".equals(primitiveName)) {
            return Byte.class;
        }
        if ("char".equals(primitiveName)) {
            return Character.class;
        }
        if ("double".equals(primitiveName)) {
            return Double.class;

        }
        if ("float".equals(primitiveName)) {
            return Float.class;
        }

        if ("int".equals(primitiveName)) {
            return Integer.class;
        }

        if ("long".equals(primitiveName)) {
            return Long.class;

        }
        if ("short".equals(primitiveName)) {
            return Short.class;

        }
        if ("void".equals(primitiveName)) {
            return Void.class;

        }
        return null;
    }

    private Class[] findConstructorArgumentTypes(Class realMpClass, Object[] constructorArguments) {
        Constructor[] declaredConstructors = realMpClass.getDeclaredConstructors();
        for (Constructor constructor : declaredConstructors) {
            Class[] parameterTypes = constructor.getParameterTypes();
            if (parameterTypes.length == constructorArguments.length) {
                boolean mapsCorrectly = true;
                for (int i = 0; i < parameterTypes.length; i++) {
                    boolean matchesArgument;
                    if (constructorArguments[i] != null) {
                        Class parameterClass = parameterTypes[i].isPrimitive() ? getPrimitiveWrapperClass(parameterTypes[i]) : parameterTypes[i];
                        Class constructorArgumentClass = constructorArguments[i].getClass().isPrimitive() ? getPrimitiveWrapperClass(constructorArguments[i].getClass()) : constructorArguments[i].getClass();

                        matchesArgument = parameterClass.isAssignableFrom(constructorArgumentClass);
                    } else {
                        matchesArgument = true;
                    }

                    mapsCorrectly &= matchesArgument;
                }
                if (mapsCorrectly) {
                    return parameterTypes;
                }
            }
        }
        return null;
    }

    protected Enhancer createEnhancer(Class realMpClass, MessageProcessorId id, Map<String, String> attributes, String fileName, String lineNumber) {

        Enhancer e = new Enhancer();
        e.setSuperclass(realMpClass);
        e.setUseCache(false);
        e.setAttemptLoad(true);
        e.setInterceptDuringConstruction(true);
        e.setNamingPolicy(new MunitNamingPolicy());

        if (FactoryBean.class.isAssignableFrom(realMpClass)) {
            createFactoryBeanCallback(id, attributes, fileName, lineNumber, e);
        } else {
            createMessageProcessorCallback(id, attributes, fileName, lineNumber, e);
        }
        return e;
    }


    private void createMessageProcessorCallback(MessageProcessorId id, Map<String, String> attributes, String fileName, String lineNumber, Enhancer e) {
        MunitMessageProcessorInterceptor callback = new MunitMessageProcessorInterceptor();
        callback.setId(id);
        callback.setAttributes(attributes);
        callback.setFileName(fileName);
        callback.setLineNumber(lineNumber);
        e.setCallbacks(new Callback[]{callback, NULL_METHOD_INTERCEPTOR});
        e.setCallbackFilter(MESSAGE_PROCESSOR_FILTER);
    }

    private void createFactoryBeanCallback(MessageProcessorId id, Map<String, String> attributes, String fileName, String lineNumber, Enhancer e) {
        MessageProcessorFactoryBeanInterceptor callback = new MessageProcessorFactoryBeanInterceptor();
        callback.setId(id);
        callback.setAttributes(attributes);
        callback.setFileName(fileName);
        callback.setLineNumber(lineNumber);
        e.setCallbacks(new Callback[]{callback, NULL_METHOD_INTERCEPTOR});
        e.setCallbackFilter(FACTORY_BEAN_FILTER);
    }

    /**
     * <p>
     * Actual implementation of the interceptor creation
     * </p>
     *
     * @return <p>
     * A {@link MunitMessageProcessorInterceptor} object
     * </p>
     */
    @Override
    protected MethodInterceptor createInterceptor() {
        return new MunitMessageProcessorInterceptor();
    }


    @Override
    public Object create(Class realMpClass, Object... objects) {
        return super.create(realMpClass, objects);
    }

    /**
     * <p>
     * Factory method used to create Message Processors without constructor parameters.
     * </p>
     *
     * @param realMpClass The class that we want to mock
     * @param id          The {@link MessageProcessorId} that identifies the message processor
     * @param attributes  The Message Processor attributes used to identify the mock
     * @param fileName    The name of the file where the message processor is written down
     * @param lineNumber  The line number where the message processor is written down
     * @return The Mocked object, if it fails mocking then the real object.
     */
    public Object create(Class realMpClass, MessageProcessorId id, Map<String, String> attributes, String fileName, String lineNumber) {
        try {
            Enhancer e = createEnhancer(realMpClass, id, attributes, fileName, lineNumber);
            return e.create();
        } catch (Throwable e) {
            logger.warn("The message processor " + id.getFullName() + " could not be mocked");
            try {
                return realMpClass.newInstance();
            } catch (Throwable e1) {
                throw new Error("The message processor " + id.getFullName() + " could not be created", e);
            }
        }
    }


    // When Spring look for a constructor it uses the type of them as well as the number of parameters based on the actual bean that needs to be created
    // As it uses reflection we can not relies on varargs for it won't match the number of parameters
    public Object create(Class realMpClass, MessageProcessorId id, Map<String, String> attributes, String fileName, String lineNumber,
                         Object... constructorArgument) {
        return doCreate(realMpClass, id, attributes, fileName, lineNumber, constructorArgument);
    }

    public Object create(Class realMpClass, MessageProcessorId id, Map<String, String> attributes, String fileName, String lineNumber,
                         Object constructorArgument) {
        return doCreate(realMpClass, id, attributes, fileName, lineNumber, new Object[]{constructorArgument});
    }

    public Object create(Class realMpClass, MessageProcessorId id, Map<String, String> attributes, String fileName, String lineNumber,
                         Object constructorArgument1, Object constructorArgument2) {
        return doCreate(realMpClass, id, attributes, fileName, lineNumber, new Object[]{constructorArgument1, constructorArgument2});
    }

    public Object create(Class realMpClass, MessageProcessorId id, Map<String, String> attributes, String fileName, String lineNumber,
                         Object constructorArgument1, Object constructorArgument2, Object constructorArgument3) {
        return doCreate(realMpClass, id, attributes, fileName, lineNumber, new Object[]{constructorArgument1, constructorArgument2, constructorArgument3});
    }

    public Object create(Class realMpClass, MessageProcessorId id, Map<String, String> attributes, String fileName, String lineNumber,
                         Object constructorArgument1, Object constructorArgument2, Object constructorArgument3, Object constructorArgument4) {
        return doCreate(realMpClass, id, attributes, fileName, lineNumber, new Object[]{constructorArgument1, constructorArgument2, constructorArgument3, constructorArgument4});
    }

    public Object create(Class realMpClass, MessageProcessorId id, Map<String, String> attributes, String fileName, String lineNumber,
                         Object constructorArgument1, Object constructorArgument2, Object constructorArgument3, Object constructorArgument4, Object constructorArgument5) {
        return doCreate(realMpClass, id, attributes, fileName, lineNumber, new Object[]{constructorArgument1, constructorArgument2, constructorArgument3, constructorArgument4, constructorArgument5});
    }

    public Object create(Class realMpClass, MessageProcessorId id, Map<String, String> attributes, String fileName, String lineNumber,
                         Object constructorArgument1, Object constructorArgument2, Object constructorArgument3, Object constructorArgument4, Object constructorArgument5, Object constructorArgument6) {
        return doCreate(realMpClass, id, attributes, fileName, lineNumber, new Object[]{constructorArgument1, constructorArgument2, constructorArgument3, constructorArgument4, constructorArgument5, constructorArgument6});
    }

    public Object create(Class realMpClass, MessageProcessorId id, Map<String, String> attributes, String fileName, String lineNumber,
                         Object constructorArgument1, Object constructorArgument2, Object constructorArgument3, Object constructorArgument4, Object constructorArgument5, Object constructorArgument6, Object constructorArgument7) {
        return doCreate(realMpClass, id, attributes, fileName, lineNumber, new Object[]{constructorArgument1, constructorArgument2, constructorArgument3, constructorArgument4, constructorArgument5, constructorArgument6, constructorArgument7});
    }

    public Object create(Class realMpClass, MessageProcessorId id, Map<String, String> attributes, String fileName, String lineNumber,
                         Object constructorArgument1, Object constructorArgument2, Object constructorArgument3, Object constructorArgument4, Object constructorArgument5, Object constructorArgument6, Object constructorArgument7, Object constructorArgument8) {
        return doCreate(realMpClass, id, attributes, fileName, lineNumber, new Object[]{constructorArgument1, constructorArgument2, constructorArgument3, constructorArgument4, constructorArgument5, constructorArgument6, constructorArgument7, constructorArgument8});
    }

    public Object create(Class realMpClass, MessageProcessorId id, Map<String, String> attributes, String fileName, String lineNumber,
                         Object constructorArgument1, Object constructorArgument2, Object constructorArgument3, Object constructorArgument4, Object constructorArgument5, Object constructorArgument6, Object constructorArgument7, Object constructorArgument8, Object constructorArgument9) {
        return doCreate(realMpClass, id, attributes, fileName, lineNumber, new Object[]{constructorArgument1, constructorArgument2, constructorArgument3, constructorArgument4, constructorArgument5, constructorArgument6, constructorArgument7, constructorArgument8, constructorArgument9});
    }

    public Object create(Class realMpClass, MessageProcessorId id, Map<String, String> attributes, String fileName, String lineNumber,
                         Object constructorArgument1, Object constructorArgument2, Object constructorArgument3, Object constructorArgument4, Object constructorArgument5, Object constructorArgument6, Object constructorArgument7, Object constructorArgument8, Object constructorArgument9, Object constructorArgument10) {
        return doCreate(realMpClass, id, attributes, fileName, lineNumber, new Object[]{constructorArgument1, constructorArgument2, constructorArgument3, constructorArgument4, constructorArgument5, constructorArgument6, constructorArgument7, constructorArgument8, constructorArgument9, constructorArgument10});
    }

    public Object create(Class realMpClass, MessageProcessorId id, Map<String, String> attributes, String fileName, String lineNumber,
                         Object constructorArgument1, Object constructorArgument2, Object constructorArgument3, Object constructorArgument4, Object constructorArgument5, Object constructorArgument6, Object constructorArgument7, Object constructorArgument8, Object constructorArgument9, Object constructorArgument10, Object constructorArgument11) {
        return doCreate(realMpClass, id, attributes, fileName, lineNumber, new Object[]{constructorArgument1, constructorArgument2, constructorArgument3, constructorArgument4, constructorArgument5, constructorArgument6, constructorArgument7, constructorArgument8, constructorArgument9, constructorArgument10, constructorArgument11});
    }

    public Object create(Class realMpClass, MessageProcessorId id, Map<String, String> attributes, String fileName, String lineNumber,
                         Object constructorArgument1, Object constructorArgument2, Object constructorArgument3, Object constructorArgument4, Object constructorArgument5, Object constructorArgument6, Object constructorArgument7, Object constructorArgument8, Object constructorArgument9, Object constructorArgument10, Object constructorArgument11, Object constructorArgument12) {
        return doCreate(realMpClass, id, attributes, fileName, lineNumber, new Object[]{constructorArgument1, constructorArgument2, constructorArgument3, constructorArgument4, constructorArgument5, constructorArgument6, constructorArgument7, constructorArgument8, constructorArgument9, constructorArgument10, constructorArgument11, constructorArgument12});
    }

    public Object create(Class realMpClass, MessageProcessorId id, Map<String, String> attributes, String fileName, String lineNumber,
                         Object constructorArgument1, Object constructorArgument2, Object constructorArgument3, Object constructorArgument4, Object constructorArgument5, Object constructorArgument6, Object constructorArgument7, Object constructorArgument8, Object constructorArgument9, Object constructorArgument10, Object constructorArgument11, Object constructorArgument12, Object constructorArgument13) {
        return doCreate(realMpClass, id, attributes, fileName, lineNumber, new Object[]{constructorArgument1, constructorArgument2, constructorArgument3, constructorArgument4, constructorArgument5, constructorArgument6, constructorArgument7, constructorArgument8, constructorArgument9, constructorArgument10, constructorArgument11, constructorArgument12, constructorArgument13});
    }


    /**
     * <p>
     * Factory method used to create Message Processors with constructor parameters.
     * </p>
     *
     * @param realMpClass          The class that we want to mock
     * @param id                   The {@link MessageProcessorId} that identifies the message processor
     * @param attributes           The Message Processor attributes used to identify the mock
     * @param fileName             The name of the file where the message processor is written down
     * @param lineNumber           The line number where the message processor is written down
     * @param constructorArguments The Array of constructor arguments of the message processor
     * @return The Mocked object, if it fails mocking then the real object.
     */
    private Object doCreate(Class realMpClass, MessageProcessorId id, Map<String, String> attributes, String fileName, String lineNumber,
                            Object[] constructorArguments) {
        try {
            Enhancer e = createEnhancer(realMpClass, id, attributes, fileName, lineNumber);
            if (constructorArguments != null && constructorArguments.length != 0) {
                Class[] classes = findConstructorArgumentTypes(realMpClass, constructorArguments);
                if (classes != null) {
                    return e.create(classes, constructorArguments);
                } else {
                    throw new Error("The message processor " + id.getFullName() + " could not be created, because " +
                            "there is no matching constructor");
                }
            } else {
                return e.create();
            }
        } catch (Throwable e) {
            logger.warn("The message processor " + id.getFullName() + " could not be mocked");
            try {
                return createRealMpInstance(realMpClass, id, constructorArguments);
            } catch (Throwable e1) {
                throw new Error("The message processor " + id.getFullName() + " could not be created", e1);
            }
        }
    }


}
