/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.common.registry;

import net.sf.cglib.proxy.Enhancer;
import org.mule.api.registry.ResolverException;
import org.mule.api.transformer.Transformer;
import org.mule.config.i18n.CoreMessages;
import org.mule.registry.TypeBasedTransformerResolver;
import org.mule.transformer.TransformerWeighting;

import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;


/**
 * Transformer resolver that takes into consideration MUnit enhanced classes.
 */
public class MUnitTypeBasedTransformerResolver extends TypeBasedTransformerResolver {

    /**
     * This method attempts to get the nearest transformer match by applying the following heuristic: it sorts the transformers by weight
     * and, from the maximum weighted ones, it tries to get the one that is enhanced by MUnit; in case of failure, it gets the not enhanced best match.
     * @param trans Transformers list.
     * @param input Input class.
     * @param output Output class.
     * @return The transformer that is the best match for the ordered pair (input,output).
     * @throws ResolverException
     */
    @Override
    protected Transformer getNearestTransformerMatch(List<Transformer> trans, Class input, Class output) throws ResolverException {
        Transformer transformer = null;
        if(trans.size() == 1) {
            transformer = trans.get(0);
        }
        if (trans.size() > 1) {
            logger.debug("Comparing transformers for best match: source = " + input + " target = " + output + " Possible transformers = " + trans);

            List<TransformerWeighting> weightings = calculateTransformerWeightings(trans, input, output);
            final TransformerWeighting maxWeighting = weightings.get(weightings.size() - 1);
            List<TransformerWeighting> maxWeightings = getMaxWeightingTransformers(weightings, maxWeighting);

            transformer = getEnhancedByMunit(maxWeightings);
            if(transformer == null) {
                transformer = maxWeightings.get(0).getTransformer();
            }
        }
        return transformer;
    }

    /**
     *
     * @param weightings Sorted list of weights.
     * @param maxWeighting Maximum weight present in the weightings list.
     * @return List of transformers with maximum weight.
     */
    private List<TransformerWeighting> getMaxWeightingTransformers(List<TransformerWeighting> weightings, TransformerWeighting maxWeighting) {
        List<TransformerWeighting> maxWeightings = new ArrayList<TransformerWeighting>();
        for(TransformerWeighting weighting : weightings) {
            if(maxWeighting.compareTo(weighting) == 0) {
                maxWeightings.add(weighting);
            }
        }
        return maxWeightings;
    }

    /**
     * Get the first enhanced transformer that can be found in the list.
     * @param weightings List of weightings.
     * @return An enhanced tranformer or null if no enhanced transformer can be found.
     */
    private Transformer getEnhancedByMunit(List<TransformerWeighting> weightings) {
        for(TransformerWeighting weighting : weightings) {
            if(Enhancer.isEnhanced(weighting.getTransformer().getClass())) {
                return weighting.getTransformer();
            }
        }
        return null;
    }

    private List<TransformerWeighting> calculateTransformerWeightings(List<Transformer> transformers, Class input, Class output)
    {
        List<TransformerWeighting> weightings = new ArrayList<TransformerWeighting>(transformers.size());

        for (Transformer transformer : transformers)
        {
            TransformerWeighting transformerWeighting = new TransformerWeighting(input, output, transformer);
            weightings.add(transformerWeighting);
        }

        Collections.sort(weightings);

        return weightings;
    }
}
