/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.common.behavior;

import org.mule.munit.common.interceptor.matchers.Matcher;
import org.mule.runtime.api.meta.NamedObject;

import java.util.HashMap;
import java.util.Map;

/**
 * Representation of the processor call for MUnit
 *
 * @author Mulesoft Inc.
 * @since 1.0.0
 */
public class ProcessorCall {

  private ProcessorId processorId;

  private Map<String, Object> attributes = new HashMap<>();

  public ProcessorCall(ProcessorId processorId) {
    this.processorId = processorId;
  }

  public ProcessorId getProcessorId() {
    return processorId;
  }

  public Map<String, Object> getAttributes() {
    return attributes;
  }

  public void setAttributes(Map<String, Object> attributes) {
    this.attributes = attributes;
  }

  /**
   * Defines if this call is of a particular processor
   *
   * @param id The id of the processor
   * @return true/false
   */
  public boolean isCallOf(ProcessorId id) {
    return this.processorId.equals(id);
  }

  /**
   * If the current call matches exactly with the call sent by parameter. Then it returns the number of attributes of the current
   * call, else returns -1
   *
   * @param call The processor call that we need to compare against.
   * @return The total number of matching attributes or -1
   */
  public int matchingWeight(ProcessorCall call) {
    if (processorId.getFullName().equals(call.getProcessorId().getFullName()) ||
        (call.getProcessorId().getName().matches(processorId.getName()) &&
            call.getProcessorId().getNamespace().matches(processorId.getNamespace()))) {

      Map<String, Object> callAttributes = call.getAttributes();
      for (Map.Entry<String, Object> myAttribute : attributes.entrySet()) {
        if (!callHasAttribute(callAttributes, myAttribute)) {
          return -1;
        }
      }

      return attributes.size();
    }
    return -1;
  }

  private boolean callHasAttribute(Map<String, Object> callAttributes, Map.Entry<String, Object> myAttribute) {
    String myAttributeKey = myAttribute.getKey();
    return callAttributes.containsKey(myAttributeKey) &&
        (matchAttributeValue(callAttributes.get(myAttributeKey), myAttribute.getValue())
            || matchAttributeName(callAttributes.get(myAttributeKey), myAttribute.getValue()));
  }

  private boolean matchAttributeValue(Object value, Object matcher) {
    if (matcher == null) {
      return value == null;
    }

    if (matcher instanceof Matcher) {
      return ((Matcher) matcher).match(value);
    }

    return matcher.equals(value);
  }

  private boolean matchAttributeName(Object value, Object matcher) {
    if (value instanceof NamedObject) {
      return matchAttributeValue(((NamedObject) value).getName(), matcher);
    }

    return false;
  }

}
